# 🔐 COMPLETE AUTH SYSTEM - Food Delivery App

## ✅ System Overview

**Authentication:** Laravel Sanctum (API Token)
**Authorization:** Spatie Laravel Permission (Roles & Permissions)
**Roles:** Admin, Manager, Cashier, Delivery Boy, Customer

---

## 📦 Installation

```bash
# Already installed
composer require laravel/sanctum
composer require spatie/laravel-permission

# Publish configs
php artisan vendor:publish --provider="Laravel\Sanctum\SanctumServiceProvider"
php artisan vendor:publish --provider="Spatie\Permission\PermissionServiceProvider"

# Run migrations
php artisan migrate
php artisan db:seed --class=RolePermissionSeeder
```

---

## 🗄️ Database Structure

### Migrations (Already Created)
1. `create_permission_tables` - Spatie package migration
2. `add_phone_to_users_table` - Custom user fields

### Tables Created
- `users` - User accounts
- `roles` - User roles
- `permissions` - System permissions
- `model_has_roles` - User-role pivot
- `model_has_permissions` - User-permission pivot
- `role_has_permissions` - Role-permission pivot
- `personal_access_tokens` - Sanctum tokens

---

## 👥 Roles & Permissions

### Roles
```php
1. admin          - Full system access
2. manager        - Restaurant management
3. cashier        - Order processing
4. delivery-boy   - Delivery management
5. customer       - Place orders
```

### Permissions
```php
- manage-users
- manage-roles
- manage-restaurants
- manage-products
- process-orders
- assign-delivery
- view-earnings
- track-delivery
```

### Role-Permission Matrix
```
Admin:        All permissions
Manager:      manage-restaurants, manage-products, process-orders, view-earnings
Cashier:      process-orders
Delivery Boy: track-delivery
Customer:     (No special permissions)
```

---

## 🔧 Implementation Files

### 1. User Model
**File:** `app/Models/User.php`

```php
<?php

namespace App\Models;

use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;
use Spatie\Permission\Traits\HasRoles;

class User extends Authenticatable
{
    use HasApiTokens, Notifiable, HasRoles;

    protected $fillable = ['name', 'email', 'password', 'phone', 'avatar'];
    protected $hidden = ['password', 'remember_token'];
    protected $casts = [
        'email_verified_at' => 'datetime',
        'password' => 'hashed',
    ];

    // Relationships
    public function restaurants() { return $this->hasMany(Restaurant::class); }
    public function orders() { return $this->hasMany(Order::class); }
    public function wallet() { return $this->hasOne(Wallet::class); }
    public function deliveryBoy() { return $this->hasOne(DeliveryBoy::class); }
}
```

---

### 2. Auth Controller
**File:** `app/Http/Controllers/API/AuthController.php`

```php
<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\ValidationException;

class AuthController extends Controller
{
    public function register(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:8|confirmed',
            'phone' => 'nullable|string|max:20',
        ]);

        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'phone' => $request->phone,
        ]);

        $user->assignRole('customer');
        $user->wallet()->create(['balance' => 0]);

        $token = $user->createToken('auth_token')->plainTextToken;

        return response()->json([
            'user' => $user->load('roles'),
            'token' => $token,
        ], 201);
    }

    public function login(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'password' => 'required',
        ]);

        $user = User::where('email', $request->email)->first();

        if (!$user || !Hash::check($request->password, $user->password)) {
            throw ValidationException::withMessages([
                'email' => ['The provided credentials are incorrect.'],
            ]);
        }

        $token = $user->createToken('auth_token')->plainTextToken;

        return response()->json([
            'user' => $user->load('roles', 'wallet'),
            'token' => $token,
        ]);
    }

    public function logout(Request $request)
    {
        $request->user()->currentAccessToken()->delete();

        return response()->json(['message' => 'Logged out successfully']);
    }

    public function user(Request $request)
    {
        return response()->json($request->user()->load('roles', 'wallet'));
    }
}
```

---

### 3. Role Middleware
**File:** `app/Http/Middleware/RoleMiddleware.php`

```php
<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;

class RoleMiddleware
{
    public function handle(Request $request, Closure $next, $role)
    {
        if (!$request->user() || !$request->user()->hasRole($role)) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        return $next($request);
    }
}
```

**Register in:** `app/Http/Kernel.php`
```php
protected $middlewareAliases = [
    // ... other middleware
    'role' => \App\Http\Middleware\RoleMiddleware::class,
];
```

---

### 4. API Routes
**File:** `routes/api.php`

```php
<?php

use Illuminate\Support\Facades\Route;
use App\Http\Controllers\API\AuthController;

// Public routes
Route::post('/register', [AuthController::class, 'register']);
Route::post('/login', [AuthController::class, 'login']);

// Protected routes
Route::middleware('auth:sanctum')->group(function () {
    Route::post('/logout', [AuthController::class, 'logout']);
    Route::get('/user', [AuthController::class, 'user']);

    // Admin only routes
    Route::middleware('role:admin')->group(function () {
        Route::get('/admin/dashboard', [AdminController::class, 'dashboard']);
    });

    // Manager only routes
    Route::middleware('role:manager')->group(function () {
        Route::post('/restaurants', [RestaurantController::class, 'store']);
    });

    // Delivery Boy only routes
    Route::middleware('role:delivery-boy')->group(function () {
        Route::get('/delivery-boy/available', [DeliveryBoyController::class, 'availableDeliveries']);
    });
});
```

---

### 5. Role Seeder
**File:** `database/seeders/RolePermissionSeeder.php`

```php
<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Spatie\Permission\Models\Role;
use Spatie\Permission\Models\Permission;

class RolePermissionSeeder extends Seeder
{
    public function run(): void
    {
        // Create Permissions
        $permissions = [
            'manage-users',
            'manage-roles',
            'manage-restaurants',
            'manage-products',
            'process-orders',
            'assign-delivery',
            'view-earnings',
            'track-delivery',
        ];

        foreach ($permissions as $permission) {
            Permission::create(['name' => $permission]);
        }

        // Create Roles
        $admin = Role::create(['name' => 'admin']);
        $manager = Role::create(['name' => 'manager']);
        $cashier = Role::create(['name' => 'cashier']);
        $deliveryBoy = Role::create(['name' => 'delivery-boy']);
        $customer = Role::create(['name' => 'customer']);

        // Assign Permissions to Roles
        $admin->givePermissionTo(Permission::all());
        $manager->givePermissionTo(['manage-restaurants', 'manage-products', 'process-orders', 'view-earnings']);
        $cashier->givePermissionTo(['process-orders']);
        $deliveryBoy->givePermissionTo(['track-delivery']);
    }
}
```

---

### 6. User Seeder
**File:** `database/seeders/UserSeeder.php`

```php
<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\User;
use Illuminate\Support\Facades\Hash;

class UserSeeder extends Seeder
{
    public function run(): void
    {
        $admin = User::create([
            'name' => 'Admin User',
            'email' => 'admin@test.com',
            'password' => Hash::make('password'),
            'phone' => '1234567890',
        ]);
        $admin->assignRole('admin');
        $admin->wallet()->create(['balance' => 10000]);

        $manager = User::create([
            'name' => 'Restaurant Manager',
            'email' => 'manager@test.com',
            'password' => Hash::make('password'),
            'phone' => '1234567891',
        ]);
        $manager->assignRole('manager');
        $manager->wallet()->create(['balance' => 5000]);

        $customer = User::create([
            'name' => 'Customer User',
            'email' => 'customer@test.com',
            'password' => Hash::make('password'),
            'phone' => '1234567892',
        ]);
        $customer->assignRole('customer');
        $customer->wallet()->create(['balance' => 1000]);

        $deliveryBoy = User::create([
            'name' => 'Delivery Boy',
            'email' => 'delivery@test.com',
            'password' => Hash::make('password'),
            'phone' => '1234567893',
        ]);
        $deliveryBoy->assignRole('delivery-boy');
        $deliveryBoy->wallet()->create(['balance' => 500]);
    }
}
```

---

## 📡 API Examples

### 1. Register (Customer)
```http
POST /api/register
Content-Type: application/json

{
  "name": "John Doe",
  "email": "john@example.com",
  "password": "password123",
  "password_confirmation": "password123",
  "phone": "1234567890"
}
```

**Response (201):**
```json
{
  "user": {
    "id": 1,
    "name": "John Doe",
    "email": "john@example.com",
    "phone": "1234567890",
    "roles": [
      {
        "id": 5,
        "name": "customer"
      }
    ]
  },
  "token": "1|abc123xyz..."
}
```

---

### 2. Login
```http
POST /api/login
Content-Type: application/json

{
  "email": "admin@test.com",
  "password": "password"
}
```

**Response (200):**
```json
{
  "user": {
    "id": 1,
    "name": "Admin User",
    "email": "admin@test.com",
    "roles": [
      {
        "id": 1,
        "name": "admin"
      }
    ],
    "wallet": {
      "balance": 10000
    }
  },
  "token": "2|xyz789abc..."
}
```

---

### 3. Get Current User
```http
GET /api/user
Authorization: Bearer {token}
```

**Response (200):**
```json
{
  "id": 1,
  "name": "Admin User",
  "email": "admin@test.com",
  "roles": [
    {
      "id": 1,
      "name": "admin",
      "permissions": [...]
    }
  ],
  "wallet": {
    "balance": 10000
  }
}
```

---

### 4. Logout
```http
POST /api/logout
Authorization: Bearer {token}
```

**Response (200):**
```json
{
  "message": "Logged out successfully"
}
```

---

### 5. Access Protected Route (Admin Only)
```http
GET /api/admin/dashboard
Authorization: Bearer {admin_token}
```

**Response (200):** Dashboard data

**Response (403) - If not admin:**
```json
{
  "message": "Unauthorized"
}
```

---

## 🔒 Security Features

### 1. Rate Limiting
```php
// In app/Http/Kernel.php
'api' => [
    'throttle:60,1', // 60 requests per minute
],
```

### 2. Token Expiration
```php
// In config/sanctum.php
'expiration' => 60 * 24, // 24 hours
```

### 3. Password Hashing
```php
// Automatic with Laravel
'password' => Hash::make($request->password)
```

---

## 🧪 Testing

### Test Users
```
Admin:        admin@test.com / password
Manager:      manager@test.com / password
Customer:     customer@test.com / password
Delivery Boy: delivery@test.com / password
```

### Using Postman
1. Register/Login to get token
2. Add header: `Authorization: Bearer {token}`
3. Test protected routes

### Using cURL
```bash
# Login
curl -X POST http://localhost:8000/api/login \
  -H "Content-Type: application/json" \
  -d '{"email":"admin@test.com","password":"password"}'

# Get user (with token)
curl -X GET http://localhost:8000/api/user \
  -H "Authorization: Bearer {token}"
```

---

## ✅ Checklist

- [x] Sanctum installed & configured
- [x] Spatie Permission installed
- [x] User model with HasRoles trait
- [x] AuthController (register, login, logout)
- [x] RoleMiddleware created
- [x] Middleware registered in Kernel
- [x] API routes protected
- [x] Role seeder created
- [x] User seeder with test accounts
- [x] Rate limiting enabled
- [x] Token authentication working

---

## 🚀 Quick Start

```bash
# 1. Run migrations
php artisan migrate

# 2. Seed roles & users
php artisan db:seed --class=RolePermissionSeeder
php artisan db:seed --class=UserSeeder

# 3. Start server
php artisan serve

# 4. Test login
POST http://localhost:8000/api/login
{
  "email": "admin@test.com",
  "password": "password"
}
```

---

## 📚 Usage in Code

### Check Role
```php
if ($user->hasRole('admin')) {
    // Admin logic
}
```

### Check Permission
```php
if ($user->can('manage-restaurants')) {
    // Allow action
}
```

### Assign Role
```php
$user->assignRole('manager');
```

### Get User Roles
```php
$roles = $user->getRoleNames(); // ['admin']
```

---

**AUTH SYSTEM COMPLETE!** 🎉
