# 🍕 Product Management System - Complete Documentation

## 📋 Overview

A complete, production-ready product management system for food delivery with role-based access control, approval workflows, and comprehensive CRUD operations.

---

## 🗄️ Database Schema

### Products Table
```sql
- id (PK)
- restaurant_id (FK)
- category_id (FK)
- name
- slug
- description
- price (base price)
- discount_price
- discount_type (percentage/fixed)
- image
- images (JSON - multiple images)
- is_available (in stock)
- is_active (enabled by manager)
- is_approved (approved by admin)
- stock_quantity
- is_featured
- is_vegetarian
- is_spicy
- preparation_time (minutes)
- sort_order
- rating
- total_reviews
- total_orders
- timestamps
- soft_deletes
```

### Product Variations Table
```sql
- id (PK)
- product_id (FK)
- name (e.g., "Size")
- value (e.g., "Large")
- price_adjustment (additional price)
- is_available
- sort_order
- timestamps
```

### Product Add-ons Table
```sql
- id (PK)
- product_id (FK)
- name (e.g., "Extra Cheese")
- price
- is_available
- sort_order
- timestamps
```

---

## 🔐 Role-Based Access Control

### Super Admin
- View all products (all restaurants)
- Approve/Reject products
- Enable/Disable products
- Delete any product
- View product analytics
- Full system control

### Admin (Restaurant Owner)
- Manage only own restaurant products
- Create/Edit/Delete products
- Set prices, discounts, images
- Manage variations & add-ons
- Control stock & availability
- Products require super-admin approval

### Manager
- Same as Admin
- Manage restaurant products
- Full CRUD operations

### Cashier
- Toggle product availability (In stock / Out of stock)
- Update stock quantity
- View-only access to products

### Customer
- View approved, active, available products only
- Cannot see pending or inactive products

---

## 🛠️ API Endpoints

### Admin Endpoints
```
GET    /api/admin/products                    - List all products (with filters)
POST   /api/admin/products/{id}/approve       - Approve product
POST   /api/admin/products/{id}/reject        - Reject product
POST   /api/admin/products/{id}/toggle-status - Enable/Disable product
DELETE /api/admin/products/{id}               - Delete product
GET    /api/admin/products/{id}/analytics     - Product analytics
```

### Manager Endpoints
```
GET    /api/manager/products                  - List own products
POST   /api/manager/products                  - Create product
PUT    /api/manager/products/{id}             - Update product
DELETE /api/manager/products/{id}             - Delete product
POST   /api/manager/products/{id}/toggle-availability - Toggle availability
POST   /api/manager/products/{id}/update-stock - Update stock
```

### Cashier Endpoints
```
POST   /api/cashier/products/{id}/toggle-availability - Toggle availability
POST   /api/cashier/products/{id}/update-stock - Update stock
```

### Public Endpoints
```
GET    /api/products                          - List public products
GET    /api/products/{id}                     - Product details
GET    /api/restaurants/{id}/products         - Restaurant products
```

---

## 📊 API Request/Response Examples

### Create Product (Manager)
**Request:**
```http
POST /api/manager/products
Content-Type: multipart/form-data

{
  "name": "Margherita Pizza",
  "category_id": 1,
  "description": "Classic Italian pizza",
  "price": 1200,
  "discount_price": 1000,
  "stock_quantity": 50,
  "preparation_time": 20,
  "image": <file>,
  "is_vegetarian": true,
  "is_spicy": false,
  "variations": [
    {"name": "Small", "price": 0},
    {"name": "Medium", "price": 200},
    {"name": "Large", "price": 400}
  ],
  "addons": [
    {"name": "Extra Cheese", "price": 150},
    {"name": "Mushrooms", "price": 100}
  ]
}
```

**Response:**
```json
{
  "message": "Product created successfully. Awaiting admin approval.",
  "data": {
    "id": 1,
    "restaurant_id": 1,
    "category_id": 1,
    "name": "Margherita Pizza",
    "slug": "margherita-pizza-1234567890",
    "price": 1200,
    "discount_price": 1000,
    "is_approved": false,
    "is_active": true,
    "is_available": true,
    "stock_quantity": 50,
    "variations": [...],
    "addons": [...]
  }
}
```

### Approve Product (Admin)
**Request:**
```http
POST /api/admin/products/1/approve
```

**Response:**
```json
{
  "message": "Product approved",
  "data": {
    "id": 1,
    "is_approved": true,
    ...
  }
}
```

---

## 💰 Price Calculation Logic

### Formula
```
Total Price = (Base Price OR Variation Price) + Sum(Add-ons) × Quantity - Discount

Example:
- Base Price: Rs 1200
- Variation (Large): +Rs 400
- Add-ons (Extra Cheese + Mushrooms): Rs 150 + Rs 100 = Rs 250
- Quantity: 2
- Discount: Rs 200

Total = ((1200 + 400) + 250) × 2 - 200
      = 1850 × 2 - 200
      = 3700 - 200
      = Rs 3500
```

---

## 🎯 Product Lifecycle

```
1. Manager creates product
   ↓
2. Product status: is_approved = false, is_active = true
   ↓
3. Admin reviews product
   ↓
4. Admin approves → is_approved = true (visible to customers)
   OR
   Admin rejects → is_approved = false (not visible)
   ↓
5. Manager can toggle is_active (enable/disable)
   ↓
6. Cashier can toggle is_available (in stock/out of stock)
   ↓
7. Customer sees product only if:
   - is_approved = true
   - is_active = true
   - is_available = true
   - stock_quantity > 0
```

---

## 🔍 Product Scopes (Laravel)

```php
// Active products
Product::active()->get();

// Approved products
Product::approved()->get();

// Available products (in stock)
Product::available()->get();

// Public products (approved + active + available)
Product::public()->get();
```

---

## 🎨 Frontend Pages

### Manager Product Page
**Route:** `/manager/products`
**Features:**
- Grid view with product cards
- Search & filter (category, status)
- Add/Edit product modal
- Image upload with preview
- Variations & add-ons management
- Toggle availability button
- Delete product
- Approval status badge

### Admin Product Management
**Route:** `/admin/products`
**Features:**
- Table view with all products
- Filter by restaurant, category, status
- Approve/Reject buttons
- Enable/Disable toggle
- Delete product
- View analytics
- Multi-status badges

---

## 🚀 Features Implemented

✅ Complete CRUD operations
✅ Role-based access control
✅ Product approval workflow
✅ Image upload & preview
✅ Product variations (sizes)
✅ Product add-ons (extras)
✅ Stock management
✅ Availability toggle
✅ Search & filter
✅ Discount pricing
✅ Product analytics
✅ Soft deletes
✅ Auto-hide out-of-stock
✅ Restaurant-based filtering
✅ Category management
✅ Preparation time tracking

---

## 📝 Usage Examples

### Manager: Create Product
1. Navigate to `/manager/products`
2. Click "Add Product"
3. Fill in product details
4. Add variations (optional)
5. Add add-ons (optional)
6. Upload image
7. Click "Save Product"
8. Product awaits admin approval

### Admin: Approve Product
1. Navigate to `/admin/products`
2. Filter by "Pending Approval"
3. Review product details
4. Click "Approve" or "Reject"
5. Product becomes visible to customers

### Cashier: Update Stock
1. View products list
2. Click "Update Stock"
3. Enter new quantity
4. Availability auto-updates

---

## 🔒 Security Features

- Role-based middleware protection
- Restaurant ownership validation
- Image upload validation (type, size)
- SQL injection prevention (Eloquent ORM)
- XSS protection (Vue escaping)
- CSRF protection (Laravel Sanctum)
- Soft deletes (data recovery)

---

## 🎯 Best Practices Implemented

- Repository pattern ready
- Service layer separation
- Request validation
- API Resources for responses
- Eloquent relationships
- Database indexing
- Image optimization
- Lazy loading
- Code reusability
- Clean architecture

---

## 📦 Dependencies

**Backend:**
- Laravel 11
- Spatie Laravel Permission
- Laravel Sanctum
- Intervention Image (optional)

**Frontend:**
- Vue 3
- Pinia
- Axios
- Tailwind CSS

---

## 🧪 Testing

```bash
# Test product creation
POST /api/manager/products

# Test approval workflow
POST /api/admin/products/1/approve

# Test public visibility
GET /api/products (should only show approved products)

# Test role restrictions
GET /api/admin/products (as customer - should fail)
```

---

## 🎉 Summary

This is a **PRODUCTION-READY** product management system with:
- ✅ Complete role-based access control
- ✅ Approval workflow
- ✅ Full CRUD operations
- ✅ Variations & add-ons
- ✅ Stock management
- ✅ Image handling
- ✅ Search & filter
- ✅ Security best practices
- ✅ Clean architecture
- ✅ Scalable design

**Ready for deployment!** 🚀
