# 🚀 Quick Start Guide - Food Delivery System

## Prerequisites Check

Before starting, ensure you have:
- ✅ PHP 8.1+ installed
- ✅ Composer installed
- ✅ Node.js 16+ installed
- ✅ MySQL 5.7+ installed
- ✅ Git installed

## Step-by-Step Setup

### 1. Database Setup (5 minutes)

```bash
# Open MySQL
mysql -u root -p

# Create database
CREATE DATABASE delivery_system;

# Create user (optional)
CREATE USER 'delivery_user'@'localhost' IDENTIFIED BY 'your_password';
GRANT ALL PRIVILEGES ON delivery_system.* TO 'delivery_user'@'localhost';
FLUSH PRIVILEGES;
EXIT;
```

### 2. Environment Configuration (2 minutes)

```bash
# Navigate to project
cd g:\project\laravel\delivery_system

# Copy environment file
copy .env.example .env

# Generate application key
php artisan key:generate
```

### 3. Configure .env File (3 minutes)

Open `.env` and update:

```env
APP_NAME="Food Delivery System"
APP_URL=http://localhost:8000

DB_CONNECTION=mysql
DB_HOST=127.0.0.1
DB_PORT=3306
DB_DATABASE=delivery_system
DB_USERNAME=root
DB_PASSWORD=your_password

# Google Maps (Get from: https://console.cloud.google.com)
GOOGLE_MAPS_API_KEY=your_google_maps_api_key
```

### 4. Install Dependencies (5 minutes)

```bash
# Install PHP dependencies
composer install

# Install Node dependencies
npm install
```

### 5. Run Migrations (1 minute)

```bash
# Run all migrations
php artisan migrate

# If you get errors, try:
php artisan migrate:fresh
```

### 6. Publish Vendor Assets (1 minute)

```bash
# Publish Spatie Permission config
php artisan vendor:publish --provider="Spatie\Permission\PermissionServiceProvider"

# Publish Sanctum config
php artisan vendor:publish --provider="Laravel\Sanctum\SanctumServiceProvider"
```

### 7. Create Storage Link (30 seconds)

```bash
php artisan storage:link
```

### 8. Start Development Servers (1 minute)

Open 2 terminal windows:

**Terminal 1 - Laravel Backend:**
```bash
php artisan serve
```
Access at: http://localhost:8000

**Terminal 2 - Vite Frontend:**
```bash
npm run dev
```
Access at: http://localhost:5173

## Verify Installation

### Check Laravel
```bash
php artisan --version
# Should show: Laravel Framework 10.x.x
```

### Check Database Connection
```bash
php artisan migrate:status
# Should show list of migrations
```

### Check Routes
```bash
php artisan route:list
# Should show available routes
```

## Next Steps After Setup

### 1. Create Roles & Permissions Seeder

Create file: `database/seeders/RolePermissionSeeder.php`

```php
<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Spatie\Permission\Models\Role;
use Spatie\Permission\Models\Permission;

class RolePermissionSeeder extends Seeder
{
    public function run(): void
    {
        // Create Permissions
        $permissions = [
            'manage-users',
            'manage-roles',
            'manage-restaurants',
            'manage-products',
            'process-orders',
            'assign-delivery',
            'view-earnings',
            'track-delivery',
        ];

        foreach ($permissions as $permission) {
            Permission::create(['name' => $permission]);
        }

        // Create Roles
        $admin = Role::create(['name' => 'admin']);
        $manager = Role::create(['name' => 'manager']);
        $cashier = Role::create(['name' => 'cashier']);
        $deliveryBoy = Role::create(['name' => 'delivery-boy']);
        $customer = Role::create(['name' => 'customer']);

        // Assign Permissions to Roles
        $admin->givePermissionTo(Permission::all());
        $manager->givePermissionTo(['manage-restaurants', 'manage-products', 'process-orders', 'view-earnings']);
        $cashier->givePermissionTo(['process-orders']);
        $deliveryBoy->givePermissionTo(['track-delivery']);
    }
}
```

### 2. Run Seeder

```bash
php artisan db:seed --class=RolePermissionSeeder
```

### 3. Create Test User

```bash
php artisan tinker
```

Then in tinker:
```php
$user = \App\Models\User::create([
    'name' => 'Admin User',
    'email' => 'admin@example.com',
    'password' => bcrypt('password'),
]);

$user->assignRole('admin');
exit
```

## Common Issues & Solutions

### Issue 1: Migration Error
```
Error: SQLSTATE[42000]: Syntax error
```
**Solution:** Check database credentials in `.env`

### Issue 2: Permission Denied
```
Error: The stream or file could not be opened
```
**Solution:**
```bash
# Windows
icacls storage /grant Users:F /T
icacls bootstrap/cache /grant Users:F /T
```

### Issue 3: Key Not Found
```
Error: No application encryption key
```
**Solution:**
```bash
php artisan key:generate
```

### Issue 4: Node Modules Error
```
Error: Cannot find module
```
**Solution:**
```bash
rm -rf node_modules
npm install
```

## Development Workflow

### Daily Development Routine

1. **Start Servers**
   ```bash
   # Terminal 1
   php artisan serve
   
   # Terminal 2
   npm run dev
   ```

2. **Make Changes**
   - Edit files in `app/`, `resources/`, etc.
   - Changes auto-reload with Vite

3. **Test Changes**
   ```bash
   php artisan test
   ```

4. **Clear Caches (if needed)**
   ```bash
   php artisan optimize:clear
   ```

## Useful Development Commands

```bash
# Create new controller
php artisan make:controller API/ControllerName

# Create new model
php artisan make:model ModelName -mfsc

# Create new migration
php artisan make:migration create_table_name

# Run specific migration
php artisan migrate --path=/database/migrations/filename.php

# Rollback last migration
php artisan migrate:rollback

# Fresh start (WARNING: Deletes all data)
php artisan migrate:fresh --seed

# List all routes
php artisan route:list

# Clear all caches
php artisan cache:clear
php artisan config:clear
php artisan route:clear
php artisan view:clear
```

## Testing API Endpoints

### Using Postman/Insomnia

1. **Register User**
   - POST: `http://localhost:8000/api/register`
   - Body: `{ "name": "Test", "email": "test@test.com", "password": "password" }`

2. **Login**
   - POST: `http://localhost:8000/api/login`
   - Body: `{ "email": "test@test.com", "password": "password" }`
   - Save the token from response

3. **Authenticated Request**
   - GET: `http://localhost:8000/api/user`
   - Header: `Authorization: Bearer {your_token}`

## Project Structure Overview

```
delivery_system/
├── app/
│   ├── Http/Controllers/    # Your controllers here
│   ├── Models/              # Your models here
│   └── Services/            # Business logic here
├── database/
│   ├── migrations/          # ✅ Already created (15 tables)
│   └── seeders/             # Create seeders here
├── routes/
│   ├── api.php             # Define API routes here
│   └── web.php             # Define web routes here
└── resources/
    ├── js/                 # Vue components here
    └── views/              # Blade templates here
```

## Getting Help

- 📖 Laravel Docs: https://laravel.com/docs
- 📖 Vue 3 Docs: https://vuejs.org
- 📖 Spatie Permission: https://spatie.be/docs/laravel-permission
- 📖 Tailwind CSS: https://tailwindcss.com/docs

## Ready to Code! 🎉

Your development environment is now set up. Start building:

1. Create your first API endpoint
2. Build your first Vue component
3. Test with Postman
4. Iterate and improve

**Happy Coding!** 🚀
