# 🚀 Food Delivery System - Setup Guide

## Tech Stack
- **Backend**: Laravel 10 + REST API
- **Frontend**: Vue 3 + Composition API
- **Build Tool**: Vite
- **Styling**: Tailwind CSS
- **Database**: MySQL
- **Authentication**: Laravel Sanctum
- **Permissions**: Spatie Laravel Permission
- **Maps**: Google Maps API
- **Real-time**: Laravel Echo + Pusher/Socket.io

## Prerequisites
- PHP >= 8.1
- Composer
- Node.js >= 16
- MySQL >= 5.7
- Redis (optional, for caching)

## Installation Steps

### 1. Install PHP Dependencies
```bash
composer install
```

### 2. Install Additional Packages
```bash
# Already installed:
# - spatie/laravel-permission

# Install additional packages:
composer require intervention/image
composer require barryvdh/laravel-dompdf
composer require pusher/pusher-php-server
```

### 3. Environment Configuration
```bash
cp .env.example .env
php artisan key:generate
```

### 4. Configure .env File
```env
APP_NAME="Food Delivery System"
APP_URL=http://localhost:8000

DB_CONNECTION=mysql
DB_HOST=127.0.0.1
DB_PORT=3306
DB_DATABASE=delivery_system
DB_USERNAME=root
DB_PASSWORD=

# Sanctum
SANCTUM_STATEFUL_DOMAINS=localhost:3000,127.0.0.1:3000

# Google Maps
GOOGLE_MAPS_API_KEY=your_google_maps_api_key

# Pusher (for real-time)
BROADCAST_DRIVER=pusher
PUSHER_APP_ID=your_app_id
PUSHER_APP_KEY=your_app_key
PUSHER_APP_SECRET=your_app_secret
PUSHER_APP_CLUSTER=mt1

# Payment Gateways
STRIPE_KEY=your_stripe_key
STRIPE_SECRET=your_stripe_secret
```

### 5. Database Setup
```bash
# Create database
mysql -u root -p
CREATE DATABASE delivery_system;
exit;

# Run migrations
php artisan migrate

# Seed database with roles and sample data
php artisan db:seed
```

### 6. Publish Vendor Assets
```bash
# Publish Spatie Permission config
php artisan vendor:publish --provider="Spatie\Permission\PermissionServiceProvider"

# Publish Sanctum config
php artisan vendor:publish --provider="Laravel\Sanctum\SanctumServiceProvider"
```

### 7. Storage Link
```bash
php artisan storage:link
```

### 8. Install Frontend Dependencies
```bash
npm install
```

### 9. Install Vue 3 & Related Packages
```bash
npm install vue@next vue-router@4 pinia axios
npm install @vueuse/core
npm install -D tailwindcss postcss autoprefixer
npm install vue-toastification@next
npm install @googlemaps/js-api-loader
```

### 10. Initialize Tailwind CSS
```bash
npx tailwindcss init -p
```

### 11. Run Development Servers
```bash
# Terminal 1 - Laravel Backend
php artisan serve

# Terminal 2 - Vite Frontend
npm run dev

# Terminal 3 - Queue Worker (for jobs)
php artisan queue:work

# Terminal 4 - WebSocket Server (optional)
php artisan websockets:serve
```

## Project Structure

```
delivery_system/
├── app/
│   ├── Http/
│   │   ├── Controllers/
│   │   │   ├── API/
│   │   │   │   ├── AuthController.php
│   │   │   │   ├── RestaurantController.php
│   │   │   │   ├── ProductController.php
│   │   │   │   ├── CartController.php
│   │   │   │   ├── OrderController.php
│   │   │   │   ├── DeliveryController.php
│   │   │   │   └── PaymentController.php
│   │   ├── Middleware/
│   │   │   ├── CheckRole.php
│   │   │   └── CheckPermission.php
│   │   ├── Requests/
│   │   └── Resources/
│   ├── Models/
│   │   ├── User.php
│   │   ├── Restaurant.php
│   │   ├── Product.php
│   │   ├── ProductVariation.php
│   │   ├── ProductAddon.php
│   │   ├── Cart.php
│   │   ├── Order.php
│   │   ├── OrderItem.php
│   │   ├── Delivery.php
│   │   ├── Payment.php
│   │   └── Review.php
│   ├── Services/
│   │   ├── AuthService.php
│   │   ├── RestaurantService.php
│   │   ├── OrderService.php
│   │   ├── DeliveryService.php
│   │   ├── PaymentService.php
│   │   └── MapService.php
│   └── Repositories/
├── database/
│   ├── migrations/
│   ├── seeders/
│   └── factories/
├── resources/
│   ├── js/
│   │   ├── components/
│   │   ├── views/
│   │   ├── stores/
│   │   ├── router/
│   │   ├── services/
│   │   └── app.js
│   └── css/
│       └── app.css
├── routes/
│   ├── api.php
│   └── web.php
└── tests/
```

## User Roles & Permissions

### Roles
1. **Admin** - Full system access
2. **Manager** - Restaurant owner/manager
3. **Cashier** - Order processing
4. **Delivery Boy** - Delivery management
5. **Customer** - Place orders

### Permissions
- manage-users
- manage-roles
- manage-restaurants
- manage-products
- process-orders
- assign-delivery
- view-earnings
- track-delivery

## API Endpoints

### Authentication
- POST `/api/register` - Register user
- POST `/api/login` - Login
- POST `/api/logout` - Logout
- GET `/api/user` - Get authenticated user

### Restaurants
- GET `/api/restaurants` - List all restaurants
- POST `/api/restaurants` - Create restaurant (Manager)
- GET `/api/restaurants/{id}` - Get restaurant details
- PUT `/api/restaurants/{id}` - Update restaurant
- DELETE `/api/restaurants/{id}` - Delete restaurant

### Products
- GET `/api/products` - List products
- POST `/api/products` - Create product
- GET `/api/products/{id}` - Get product details
- PUT `/api/products/{id}` - Update product
- DELETE `/api/products/{id}` - Delete product

### Cart
- GET `/api/cart` - Get user cart
- POST `/api/cart` - Add to cart
- PUT `/api/cart/{id}` - Update cart item
- DELETE `/api/cart/{id}` - Remove from cart

### Orders
- GET `/api/orders` - List orders
- POST `/api/orders` - Create order
- GET `/api/orders/{id}` - Get order details
- PUT `/api/orders/{id}/status` - Update order status
- POST `/api/orders/{id}/cancel` - Cancel order

### Delivery
- GET `/api/deliveries` - List deliveries
- POST `/api/deliveries/{id}/accept` - Accept delivery
- PUT `/api/deliveries/{id}/status` - Update delivery status
- GET `/api/deliveries/{id}/track` - Track delivery

### Payments
- POST `/api/payments` - Process payment
- GET `/api/payments/{id}` - Get payment details

## Testing

```bash
# Run all tests
php artisan test

# Run specific test
php artisan test --filter=AuthTest
```

## Deployment

### Production Checklist
- [ ] Set `APP_ENV=production`
- [ ] Set `APP_DEBUG=false`
- [ ] Configure proper database credentials
- [ ] Set up SSL certificate
- [ ] Configure queue workers with Supervisor
- [ ] Set up Redis for caching
- [ ] Configure cloud storage (S3/DigitalOcean Spaces)
- [ ] Set up monitoring (Sentry/Bugsnag)
- [ ] Configure backup system
- [ ] Set up CDN for static assets

## Support

For issues and questions, please create an issue in the repository.

## License

This project is proprietary software.
