<?php

namespace App\Http\Controllers\API\Admin;

use App\Http\Controllers\Controller;
use App\Models\DeliveryBoy;
use App\Models\User;
use App\Models\CommissionRule;
use App\Models\Payout;
use App\Http\Resources\DeliveryBoyResource;
use App\Http\Requests\Admin\CreateDeliveryBoyRequest;
use App\Http\Requests\Admin\UpdateDeliveryBoyRequest;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;

class DeliveryBoyManagementController extends Controller
{
    /**
     * Display a listing of all delivery boys.
     */
    public function index(Request $request): JsonResponse
    {
        $query = DeliveryBoy::with(['user', 'commissionRule'])
            ->withCount(['deliveries', 'deliveries as completed_deliveries' => function($q) {
                $q->where('status', 'completed');
            }])
            ->when($request->search, function($q) use ($request) {
                $q->whereHas('user', function($query) use ($request) {
                    $query->where('name', 'like', "%{$request->search}%")
                          ->orWhere('email', 'like', "%{$request->search}%")
                          ->orWhere('phone', 'like', "%{$request->search}%");
                });
            })
            ->when($request->status, function($q) use ($request) {
                $q->where('status', $request->status);
            })
            ->when($request->is_active !== null, function($q) use ($request) {
                $q->where('is_active', $request->is_active);
            })
            ->when($request->is_verified !== null, function($q) use ($request) {
                $q->where('is_verified', $request->is_verified);
            })
            ->when($request->sort_by, function($q) use ($request) {
                $q->orderBy($request->sort_by, $request->sort_order ?? 'desc');
            }, function($q) {
                $q->orderBy('created_at', 'desc');
            });

        $deliveryBoys = $query->paginate($request->per_page ?? 20);

        return response()->json([
            'success' => true,
            'data' => DeliveryBoyResource::collection($deliveryBoys),
            'meta' => [
                'current_page' => $deliveryBoys->currentPage(),
                'last_page' => $deliveryBoys->lastPage(),
                'per_page' => $deliveryBoys->perPage(),
                'total' => $deliveryBoys->total(),
            ]
        ]);
    }

    /**
     * Store a newly created delivery boy.
     */
    public function store(CreateDeliveryBoyRequest $request): JsonResponse
    {
        return DB::transaction(function () use ($request) {
            // Create user account
            $user = User::create([
                'name' => $request->name,
                'email' => $request->email,
                'phone' => $request->phone,
                'password' => Hash::make($request->password),
            ]);

            // Assign delivery boy role
            $user->assignRole('delivery-boy');

            // Create delivery boy profile
            $deliveryBoyData = [
                'user_id' => $user->id,
                'vehicle_type' => $request->vehicle_type,
                'vehicle_number' => $request->vehicle_number,
                'license_number' => $request->license_number,
                'commission_percentage' => $request->commission_percentage ?? 10,
                'is_active' => $request->is_active ?? true,
            ];

            // Handle license image upload
            if ($request->hasFile('license_image')) {
                $deliveryBoyData['license_image'] = $request->file('license_image')->store('delivery-boys/licenses', 'public');
            }

            $deliveryBoy = DeliveryBoy::create($deliveryBoyData);

            // Create commission rule
            $deliveryBoy->commissionRule()->create([
                'name' => 'Delivery Boy Commission',
                'type' => 'percentage',
                'value' => $request->commission_percentage ?? 10,
                'status' => 'active'
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Delivery boy created successfully',
                'data' => new DeliveryBoyResource($deliveryBoy->load(['user', 'commissionRule']))
            ], 201);
        });
    }

    /**
     * Display the specified delivery boy.
     */
    public function show(DeliveryBoy $deliveryBoy): JsonResponse
    {
        $deliveryBoy->load(['user', 'commissionRule', 'deliveries'])
                   ->loadCount(['deliveries', 'deliveries as completed_deliveries' => function($q) {
                       $q->where('status', 'completed');
                   }]);

        return response()->json([
            'success' => true,
            'data' => new DeliveryBoyResource($deliveryBoy)
        ]);
    }

    /**
     * Update the specified delivery boy.
     */
    public function update(UpdateDeliveryBoyRequest $request, DeliveryBoy $deliveryBoy): JsonResponse
    {
        return DB::transaction(function () use ($request, $deliveryBoy) {
            $validated = $request->validated();

            // Update user information
            if ($request->hasAny(['name', 'email', 'phone'])) {
                $userData = [];
                if ($request->has('name')) $userData['name'] = $request->name;
                if ($request->has('email')) $userData['email'] = $request->email;
                if ($request->has('phone')) $userData['phone'] = $request->phone;
                
                $deliveryBoy->user()->update($userData);
            }

            // Handle license image upload
            if ($request->hasFile('license_image')) {
                if ($deliveryBoy->license_image) {
                    Storage::disk('public')->delete($deliveryBoy->license_image);
                }
                $validated['license_image'] = $request->file('license_image')->store('delivery-boys/licenses', 'public');
            }

            $deliveryBoy->update($validated);

            // Update commission rule if provided
            if ($request->has('commission_percentage')) {
                $deliveryBoy->commissionRule()->updateOrCreate(
                    ['commissionable_id' => $deliveryBoy->id, 'commissionable_type' => DeliveryBoy::class],
                    [
                        'name' => 'Delivery Boy Commission',
                        'type' => 'percentage',
                        'value' => $request->commission_percentage,
                        'status' => 'active'
                    ]
                );
            }

            return response()->json([
                'success' => true,
                'message' => 'Delivery boy updated successfully',
                'data' => new DeliveryBoyResource($deliveryBoy->fresh(['user', 'commissionRule']))
            ]);
        });
    }

    /**
     * Remove the specified delivery boy.
     */
    public function destroy(DeliveryBoy $deliveryBoy): JsonResponse
    {
        return DB::transaction(function () use ($deliveryBoy) {
            // Delete license image
            if ($deliveryBoy->license_image) {
                Storage::disk('public')->delete($deliveryBoy->license_image);
            }

            // Delete user account
            $deliveryBoy->user()->delete();

            // Delete delivery boy profile
            $deliveryBoy->delete();

            return response()->json([
                'success' => true,
                'message' => 'Delivery boy deleted successfully'
            ]);
        });
    }

    /**
     * Activate/suspend delivery boy.
     */
    public function toggleStatus(DeliveryBoy $deliveryBoy): JsonResponse
    {
        $deliveryBoy->update(['is_active' => !$deliveryBoy->is_active]);

        // If deactivating, set status to offline
        if (!$deliveryBoy->is_active) {
            $deliveryBoy->update(['status' => 'offline']);
        }

        return response()->json([
            'success' => true,
            'message' => 'Delivery boy status updated successfully',
            'data' => new DeliveryBoyResource($deliveryBoy)
        ]);
    }

    /**
     * Verify/unverify delivery boy.
     */
    public function toggleVerification(DeliveryBoy $deliveryBoy): JsonResponse
    {
        $deliveryBoy->update(['is_verified' => !$deliveryBoy->is_verified]);

        return response()->json([
            'success' => true,
            'message' => 'Delivery boy verification status updated successfully',
            'data' => new DeliveryBoyResource($deliveryBoy)
        ]);
    }

    /**
     * Manually assign delivery boy to order.
     */
    public function assignToOrder(DeliveryBoy $deliveryBoy, Request $request): JsonResponse
    {
        $request->validate([
            'order_id' => 'required|exists:orders,id'
        ]);

        // Check if delivery boy is available
        if ($deliveryBoy->status !== 'available' || !$deliveryBoy->is_active) {
            return response()->json([
                'success' => false,
                'message' => 'Delivery boy is not available for assignment'
            ], 422);
        }

        // Create delivery assignment logic would go here
        // This would typically create a Delivery record

        return response()->json([
            'success' => true,
            'message' => 'Delivery boy assigned to order successfully'
        ]);
    }

    /**
     * Get delivery boy earnings and history.
     */
    public function earnings(DeliveryBoy $deliveryBoy, Request $request): JsonResponse
    {
        $startDate = $request->start_date ?? now()->subDays(30);
        $endDate = $request->end_date ?? now();

        $earnings = [
            'total_earnings' => $deliveryBoy->total_earnings,
            'commission_percentage' => $deliveryBoy->commission_percentage,
            'total_deliveries' => $deliveryBoy->total_deliveries,
            'completed_deliveries' => $deliveryBoy->deliveries()->where('status', 'completed')->count(),
            'rating' => $deliveryBoy->rating,
            'total_reviews' => $deliveryBoy->total_reviews,
        ];

        // Monthly earnings breakdown
        $monthlyEarnings = $deliveryBoy->deliveries()
            ->whereBetween('created_at', [$startDate, $endDate])
            ->where('status', 'completed')
            ->selectRaw('DATE_FORMAT(created_at, "%Y-%m") as month, COUNT(*) as deliveries, SUM(delivery_fee) as earnings')
            ->groupBy('month')
            ->orderBy('month')
            ->get();

        $earnings['monthly_breakdown'] = $monthlyEarnings;

        return response()->json([
            'success' => true,
            'data' => $earnings
        ]);
    }

    /**
     * Get delivery boy payout history.
     */
    public function payouts(DeliveryBoy $deliveryBoy, Request $request): JsonResponse
    {
        $payouts = Payout::where('payable_id', $deliveryBoy->id)
            ->where('payable_type', DeliveryBoy::class)
            ->orderBy('created_at', 'desc')
            ->paginate($request->per_page ?? 20);

        return response()->json([
            'success' => true,
            'data' => $payouts
        ]);
    }

    /**
     * Process payout for delivery boy.
     */
    public function processPayout(DeliveryBoy $deliveryBoy, Request $request): JsonResponse
    {
        $request->validate([
            'amount' => 'required|numeric|min:1|max:' . $deliveryBoy->total_earnings,
            'payment_method' => 'required|string|max:50',
            'notes' => 'nullable|string|max:500'
        ]);

        return DB::transaction(function () use ($request, $deliveryBoy) {
            // Create payout record
            $payout = Payout::create([
                'user_id' => $deliveryBoy->user_id,
                'payable_id' => $deliveryBoy->id,
                'payable_type' => DeliveryBoy::class,
                'amount' => $request->amount,
                'type' => 'delivery_commission',
                'status' => 'processing',
                'payment_method' => $request->payment_method,
                'notes' => $request->notes
            ]);

            // Update delivery boy earnings
            $deliveryBoy->decrement('total_earnings', $request->amount);

            // Process payment (integration with payment gateway would go here)
            // For now, we'll mark as paid immediately
            $payout->update([
                'status' => 'paid',
                'paid_at' => now()
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Payout processed successfully',
                'data' => $payout
            ]);
        });
    }

    /**
     * Get delivery boy analytics.
     */
    public function analytics(DeliveryBoy $deliveryBoy, Request $request): JsonResponse
    {
        $startDate = $request->start_date ?? now()->subDays(30);
        $endDate = $request->end_date ?? now();

        $analytics = [
            'total_deliveries' => $deliveryBoy->total_deliveries,
            'completed_deliveries' => $deliveryBoy->deliveries()->where('status', 'completed')->count(),
            'cancelled_deliveries' => $deliveryBoy->deliveries()->where('status', 'cancelled')->count(),
            'total_earnings' => $deliveryBoy->total_earnings,
            'rating' => $deliveryBoy->rating,
            'total_reviews' => $deliveryBoy->total_reviews,
            'is_verified' => $deliveryBoy->is_verified,
            'is_active' => $deliveryBoy->is_active,
            'current_status' => $deliveryBoy->status,
        ];

        // Daily delivery stats
        $dailyStats = $deliveryBoy->deliveries()
            ->whereBetween('created_at', [$startDate, $endDate])
            ->selectRaw('DATE(created_at) as date, 
                         COUNT(*) as total_deliveries, 
                         SUM(CASE WHEN status = "completed" THEN 1 ELSE 0 END) as completed,
                         SUM(CASE WHEN status = "cancelled" THEN 1 ELSE 0 END) as cancelled,
                         SUM(delivery_fee) as earnings')
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        $analytics['daily_stats'] = $dailyStats;

        return response()->json([
            'success' => true,
            'data' => $analytics
        ]);
    }
}