<?php

namespace App\Http\Controllers\API\Admin;

use App\Http\Controllers\Controller;
use App\Models\Restaurant;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class RestaurantPromotionController extends Controller
{
    public function getRestaurantProducts($restaurantId)
    {
        try {
            $restaurant = Restaurant::findOrFail($restaurantId);
            $products = $restaurant->products()->with('categories')->get();
            
            return response()->json([
                'success' => true,
                'data' => $products
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }

    public function addProduct(Request $request, $restaurantId, $productId)
    {
        try {
            $request->validate([
                'price' => 'required|numeric|min:0',
                'stock_quantity' => 'nullable|integer|min:0',
                'preparation_time' => 'nullable|integer|min:0',
                'discount_percentage' => 'nullable|numeric|min:0|max:100',
                'is_available' => 'boolean'
            ]);

            $restaurant = Restaurant::findOrFail($restaurantId);
            
            $restaurant->products()->attach($productId, [
                'price' => $request->price,
                'stock_quantity' => $request->stock_quantity ?? 0,
                'preparation_time' => $request->preparation_time ?? 0,
                'discount_percentage' => $request->discount_percentage ?? 0,
                'is_available' => $request->boolean('is_available', true)
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Product added to restaurant successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }

    public function updatePromotion(Request $request, $restaurantId, $productId)
    {
        try {
            $request->validate([
                'price' => 'required|numeric|min:0',
                'discount_percentage' => 'nullable|numeric|min:0|max:100',
                'special_start_date' => 'nullable|date',
                'special_end_date' => 'nullable|date|after_or_equal:special_start_date',
                'preparation_time' => 'nullable|integer|min:0',
                'stock_quantity' => 'nullable|integer|min:0',
                'is_featured' => 'boolean',
                'is_available' => 'boolean',
                'special_notes' => 'nullable|string',
                'custom_image' => 'nullable|image|max:2048'
            ]);

            $restaurant = Restaurant::findOrFail($restaurantId);
            
            $data = [
                'price' => $request->price,
                'discount_percentage' => $request->discount_percentage ?? 0,
                'special_start_date' => $request->special_start_date,
                'special_end_date' => $request->special_end_date,
                'preparation_time' => $request->preparation_time ?? 0,
                'stock_quantity' => $request->stock_quantity ?? 0,
                'is_featured' => $request->boolean('is_featured'),
                'is_available' => $request->boolean('is_available'),
                'is_spicy' => $request->boolean('is_spicy'),
                'is_vegetarian' => $request->boolean('is_vegetarian'),
                'special_notes' => $request->special_notes
            ];

            if ($request->hasFile('custom_image')) {
                $pivot = $restaurant->products()->where('product_id', $productId)->first();
                if ($pivot && $pivot->pivot->custom_image) {
                    Storage::disk('public')->delete($pivot->pivot->custom_image);
                }
                $path = $request->file('custom_image')->store('promotions', 'public');
                $data['custom_image'] = $path;
            }

            $restaurant->products()->updateExistingPivot($productId, $data);

            return response()->json([
                'success' => true,
                'message' => 'Promotion updated successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }
}
