<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\DeliveryBoy;
use App\Models\Delivery;
use Illuminate\Http\Request;

class DeliveryBoyController extends Controller
{
    public function dashboard(Request $request)
    {
        $deliveryBoy = DeliveryBoy::where('user_id', $request->user()->id)->firstOrFail();
        
        $todayDeliveries = Delivery::where('delivery_boy_id', $deliveryBoy->id)
            ->whereDate('created_at', today())
            ->count();
        
        $completedToday = Delivery::where('delivery_boy_id', $deliveryBoy->id)
            ->where('status', 'delivered')
            ->whereDate('delivered_at', today())
            ->count();
        
        $pendingToday = Delivery::where('delivery_boy_id', $deliveryBoy->id)
            ->whereIn('status', ['assigned', 'picked_up', 'on_the_way'])
            ->whereDate('created_at', today())
            ->count();
        
        $todayEarnings = Delivery::where('delivery_boy_id', $deliveryBoy->id)
            ->where('status', 'delivered')
            ->whereDate('delivered_at', today())
            ->sum('delivery_boy_earning');

        $availableDeliveries = Delivery::with(['order.restaurant'])
            ->where('status', 'pending')
            ->count();

        $recentDeliveries = Delivery::with(['order.restaurant', 'order.user'])
            ->where('delivery_boy_id', $deliveryBoy->id)
            ->latest()
            ->limit(10)
            ->get();

        return response()->json([
            'stats' => [
                'today_deliveries' => $todayDeliveries,
                'completed_today' => $completedToday,
                'pending_today' => $pendingToday,
                'today_earnings' => $todayEarnings,
                'total_earnings' => $deliveryBoy->total_earnings,
                'total_deliveries' => $deliveryBoy->total_deliveries,
                'available_deliveries' => $availableDeliveries,
            ],
            'recent_deliveries' => $recentDeliveries,
        ]);
    }

    public function updateStatus(Request $request)
    {
        $deliveryBoy = DeliveryBoy::where('user_id', $request->user()->id)->firstOrFail();
        
        $deliveryBoy->update([
            'status' => $request->status,
            'current_latitude' => $request->latitude,
            'current_longitude' => $request->longitude,
        ]);

        return response()->json($deliveryBoy);
    }

    public function availableDeliveries()
    {
        $deliveries = Delivery::with(['order.restaurant'])
            ->where('status', 'pending')
            ->orWhere('status', 'assigned')
            ->get();

        return response()->json($deliveries);
    }

    public function myDeliveries(Request $request)
    {
        $deliveryBoy = DeliveryBoy::where('user_id', $request->user()->id)->firstOrFail();
        
        $deliveries = Delivery::with(['order.restaurant', 'order.user'])
            ->where('delivery_boy_id', $deliveryBoy->id)
            ->latest()
            ->paginate(20);

        return response()->json($deliveries);
    }

    public function earnings(Request $request)
    {
        $deliveryBoy = DeliveryBoy::where('user_id', $request->user()->id)->firstOrFail();
        
        $todayEarnings = Delivery::where('delivery_boy_id', $deliveryBoy->id)
            ->where('status', 'delivered')
            ->whereDate('delivered_at', today())
            ->sum('delivery_boy_earning');

        $totalEarnings = $deliveryBoy->total_earnings;
        $totalDeliveries = $deliveryBoy->total_deliveries;

        return response()->json([
            'today_earnings' => $todayEarnings,
            'total_earnings' => $totalEarnings,
            'total_deliveries' => $totalDeliveries,
        ]);
    }

    public function calculateCommission($deliveryFee, $commissionPercentage)
    {
        $deliveryBoyEarning = $deliveryFee * ($commissionPercentage / 100);
        $platformCommission = $deliveryFee - $deliveryBoyEarning;

        return [
            'delivery_boy_earning' => $deliveryBoyEarning,
            'platform_commission' => $platformCommission,
        ];
    }
}
