<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\DeliveryBoy;
use Illuminate\Http\Request;

class DeliveryBoyManagementController extends Controller
{
    public function index()
    {
        $deliveryBoys = DeliveryBoy::with(['user'])
            ->latest()
            ->paginate(20);

        return response()->json($deliveryBoys);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'user_id' => 'required|exists:users,id',
            'vehicle_type' => 'required|string|in:bike,car,scooter',
            'vehicle_number' => 'required|string|max:50',
            'license_number' => 'required|string|max:50',
            'phone' => 'required|string|max:20',
            'commission_rate' => 'required|numeric|min:0|max:100',
        ]);

        $deliveryBoy = DeliveryBoy::create($validated + [
            'status' => 'pending',
            'total_earnings' => 0,
            'total_deliveries' => 0,
        ]);

        return response()->json($deliveryBoy, 201);
    }

    public function show(DeliveryBoy $deliveryBoy)
    {
        return response()->json($deliveryBoy->load(['user', 'deliveries']));
    }

    public function update(Request $request, DeliveryBoy $deliveryBoy)
    {
        $validated = $request->validate([
            'vehicle_type' => 'sometimes|string|in:bike,car,scooter',
            'vehicle_number' => 'sometimes|string|max:50',
            'license_number' => 'sometimes|string|max:50',
            'phone' => 'sometimes|string|max:20',
            'commission_rate' => 'sometimes|numeric|min:0|max:100',
            'status' => 'sometimes|string|in:pending,approved,suspended,available,busy,offline',
        ]);

        $deliveryBoy->update($validated);

        return response()->json($deliveryBoy);
    }

    public function destroy(DeliveryBoy $deliveryBoy)
    {
        $deliveryBoy->delete();
        return response()->json(null, 204);
    }

    public function approve(DeliveryBoy $deliveryBoy)
    {
        $deliveryBoy->update(['status' => 'available']);
        return response()->json($deliveryBoy);
    }

    public function suspend(DeliveryBoy $deliveryBoy)
    {
        $deliveryBoy->update(['status' => 'suspended']);
        return response()->json($deliveryBoy);
    }

    public function restore(DeliveryBoy $deliveryBoy)
    {
        $deliveryBoy->update(['status' => 'available']);
        return response()->json($deliveryBoy);
    }

    public function analytics(DeliveryBoy $deliveryBoy)
    {
        return response()->json([
            'total_deliveries' => $deliveryBoy->total_deliveries,
            'total_earnings' => $deliveryBoy->total_earnings,
            'average_rating' => $deliveryBoy->average_rating ?? 0,
            'on_time_rate' => $deliveryBoy->on_time_rate ?? 0,
        ]);
    }

    public function deliveries(DeliveryBoy $deliveryBoy)
    {
        $deliveries = $deliveryBoy->deliveries()
            ->with(['order.restaurant', 'order.user'])
            ->latest()
            ->paginate(20);

        return response()->json($deliveries);
    }

    public function updateCommission(Request $request, DeliveryBoy $deliveryBoy)
    {
        $validated = $request->validate([
            'commission_rate' => 'required|numeric|min:0|max:100',
        ]);

        $deliveryBoy->update(['commission_rate' => $validated['commission_rate']]);

        return response()->json($deliveryBoy);
    }

    public function processPayout(Request $request, DeliveryBoy $deliveryBoy)
    {
        $validated = $request->validate([
            'amount' => 'required|numeric|min:0|max:' . $deliveryBoy->total_earnings,
            'payment_method' => 'required|string|in:bank_transfer,paypal,cash',
        ]);

        // Payout processing logic would go here
        // For now, just return success
        return response()->json([
            'message' => 'Payout processed successfully',
            'amount' => $validated['amount'],
            'delivery_boy' => $deliveryBoy,
        ]);
    }
}