<?php

namespace App\Http\Controllers\API\Manager;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\DeliveryBoy;
use App\Models\Restaurant;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class OrderController extends Controller
{
    public function index()
    {
        $user = Auth::user();
        $restaurantId = $user->restaurant_id;

        if (!$restaurantId) {
            return response()->json(['message' => 'No restaurant assigned'], 403);
        }

        $orders = Order::with(['user', 'items.product', 'restaurant', 'delivery.deliveryBoy'])
            ->where('restaurant_id', $restaurantId)
            ->orderBy('created_at', 'desc')
            ->get();

        return response()->json(['data' => $orders]);
    }

    public function show($id)
    {
        $user = Auth::user();
        $order = Order::with(['user', 'items.product', 'restaurant', 'delivery'])
            ->where('restaurant_id', $user->restaurant_id)
            ->findOrFail($id);

        return response()->json(['data' => $order]);
    }

    public function notifyDeliveryBoys($id)
    {
        $user = Auth::user();
        $order = Order::where('restaurant_id', $user->restaurant_id)->findOrFail($id);
        
        // Get restaurant
        $restaurant = Restaurant::find($user->restaurant_id);
        
        if (!$restaurant) {
            return response()->json(['message' => 'Restaurant not found'], 404);
        }

        // Find available delivery boys in same city, sorted by distance
        $deliveryBoys = DeliveryBoy::with('user')
            ->where('is_available', true)
            ->where('city', $restaurant->city)
            ->whereNotNull('latitude')
            ->whereNotNull('longitude')
            ->get()
            ->map(function ($boy) use ($restaurant) {
                $distance = $this->calculateDistance(
                    $restaurant->latitude,
                    $restaurant->longitude,
                    $boy->latitude,
                    $boy->longitude
                );
                $boy->distance = $distance;
                return $boy;
            })
            ->sortBy('distance');

        if ($deliveryBoys->isEmpty()) {
            return response()->json([
                'message' => 'No delivery boys available in ' . $restaurant->city,
                'count' => 0
            ], 404);
        }

        // Send notification to all delivery boys in same city
        foreach ($deliveryBoys as $boy) {
            // Create delivery notification
            DB::table('delivery_notifications')->insert([
                'order_id' => $order->id,
                'delivery_boy_id' => $boy->id,
                'distance' => $boy->distance,
                'status' => 'pending',
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            // Create notification for delivery boy
            \App\Models\User::find($boy->user_id)->notify(
                new \App\Notifications\NewDeliveryAssignment($order)
            );
        }

        return response()->json([
            'success' => true,
            'message' => 'Notification sent to ' . $deliveryBoys->count() . ' delivery boys',
            'count' => $deliveryBoys->count()
        ]);
    }
    
    public function updateStatus(Request $request, $id)
    {
        $user = Auth::user();
        $order = Order::where('restaurant_id', $user->restaurant_id)->findOrFail($id);
        
        $request->validate([
            'status' => 'required|in:pending,accepted,preparing,ready,picked_up,delivered,cancelled'
        ]);
        
        $order->update(['status' => $request->status]);
        
        // Auto-notify delivery boys when order is ready
        if ($request->status === 'ready') {
            $this->notifyDeliveryBoys($id);
        }
        
        return response()->json([
            'success' => true,
            'message' => 'Order status updated',
            'order' => $order->load('user', 'items.product', 'restaurant', 'delivery.deliveryBoy')
        ]);
    }
    
    private function calculateDistance($lat1, $lon1, $lat2, $lon2)
    {
        $earthRadius = 6371;
        $dLat = deg2rad($lat2 - $lat1);
        $dLon = deg2rad($lon2 - $lon1);
        $a = sin($dLat/2) * sin($dLat/2) + cos(deg2rad($lat1)) * cos(deg2rad($lat2)) * sin($dLon/2) * sin($dLon/2);
        $c = 2 * atan2(sqrt($a), sqrt(1-$a));
        return round($earthRadius * $c, 2);
    }
    
    public function history(Request $request)
    {
        $user = Auth::user();
        $restaurantId = $user->restaurant_id;

        if (!$restaurantId) {
            return response()->json(['message' => 'No restaurant assigned'], 403);
        }

        $query = Order::with(['user', 'items.product', 'restaurant', 'delivery.deliveryBoy'])
            ->where('restaurant_id', $restaurantId)
            ->whereIn('status', ['delivered', 'cancelled'])
            ->whereDate('created_at', '<', now()->toDateString());

        if ($request->search) {
            $query->where(function($q) use ($request) {
                $q->where('order_number', 'like', '%' . $request->search . '%')
                  ->orWhereHas('user', function($q) use ($request) {
                      $q->where('name', 'like', '%' . $request->search . '%');
                  });
            });
        }

        if ($request->status) {
            $query->where('status', $request->status);
        }
        
        if ($request->month && $request->year) {
            $query->whereYear('created_at', $request->year)
                  ->whereMonth('created_at', $request->month);
        }

        $orders = $query->orderBy('created_at', 'desc')->paginate(20);

        return response()->json(['data' => $orders]);
    }
    
    public function cancelledOrders(Request $request)
    {
        $user = Auth::user();
        $restaurantId = $user->restaurant_id;

        if (!$restaurantId) {
            return response()->json(['message' => 'No restaurant assigned'], 403);
        }

        $query = Order::with(['user'])
            ->where('restaurant_id', $restaurantId)
            ->where('status', 'cancelled');

        if ($request->search) {
            $query->where(function($q) use ($request) {
                $q->where('order_number', 'like', '%' . $request->search . '%')
                  ->orWhereHas('user', function($q) use ($request) {
                      $q->where('name', 'like', '%' . $request->search . '%');
                  });
            });
        }
        
        if ($request->month && $request->year) {
            $query->whereYear('created_at', $request->year)
                  ->whereMonth('created_at', $request->month);
        }

        $orders = $query->orderBy('created_at', 'desc')->paginate(20);

        return response()->json(['data' => $orders]);
    }
}
