<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\PromoCode;
use Illuminate\Http\Request;

class PromoCodeController extends Controller
{
    public function validateCode(Request $request)
    {
        $validated = $request->validate([
            'code' => 'required|string',
            'order_total' => 'required|numeric|min:0',
        ]);

        $promoCode = PromoCode::where('code', $validated['code'])
            ->where('is_active', true)
            ->where('valid_from', '<=', now())
            ->where('valid_until', '>=', now())
            ->first();

        if (!$promoCode) {
            return response()->json(['error' => 'Invalid or expired promo code'], 400);
        }

        if ($validated['order_total'] < $promoCode->minimum_order_amount) {
            return response()->json(['error' => 'Order total is below minimum requirement'], 400);
        }

        return response()->json([
            'promo_code' => $promoCode,
            'discount_amount' => $promoCode->discount_amount,
            'discount_percentage' => $promoCode->discount_percentage,
        ]);
    }

    public function apply(Request $request)
    {
        $validated = $request->validate([
            'code' => 'required|string',
            'order_total' => 'required|numeric|min:0',
        ]);

        $promoCode = PromoCode::where('code', $validated['code'])
            ->where('is_active', true)
            ->where('valid_from', '<=', now())
            ->where('valid_until', '>=', now())
            ->first();

        if (!$promoCode) {
            return response()->json(['error' => 'Invalid or expired promo code'], 400);
        }

        if ($validated['order_total'] < $promoCode->minimum_order_amount) {
            return response()->json(['error' => 'Order total is below minimum requirement'], 400);
        }

        $discount = 0;
        if ($promoCode->discount_percentage) {
            $discount = $validated['order_total'] * ($promoCode->discount_percentage / 100);
        } else {
            $discount = $promoCode->discount_amount;
        }

        return response()->json([
            'discount' => $discount,
            'final_total' => $validated['order_total'] - $discount,
        ]);
    }
}