<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\Restaurant;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class RestaurantController extends Controller
{
    public function index(Request $request)
    {
        $query = Restaurant::with('user')->where('status', 'approved');

        if ($request->search) {
            $query->where('name', 'like', "%{$request->search}%");
        }

        if ($request->is_featured) {
            $query->where('is_featured', true);
        }

        if ($request->category_id) {
            $query->whereHas('productsNew', function($q) use ($request) {
                $q->where('category_id', $request->category_id);
            });
        }

        $restaurants = $query->paginate(12);

        // Transform data
        $restaurants->getCollection()->transform(function($restaurant) {
            return [
                'id' => $restaurant->id,
                'name' => $restaurant->name,
                'description' => $restaurant->description,
                'image' => $restaurant->cover_image,
                'rating' => $restaurant->rating ?? 4.5,
                'total_reviews' => $restaurant->total_reviews ?? 0,
                'is_open' => $restaurant->operational_status === 'open',
                'is_featured' => $restaurant->is_featured,
                'delivery_time' => $restaurant->estimated_delivery_time ?? 30,
                'delivery_fee' => $restaurant->delivery_fee ?? 0,
                'minimum_order' => $restaurant->minimum_order ?? 0,
                'delivery_radius' => $restaurant->delivery_radius ?? 5,
                'distance' => null,
                'tags' => [],
                'discount' => null,
            ];
        });

        return response()->json($restaurants);
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'phone' => 'required|string|max:20',
            'email' => 'nullable|email',
            'address' => 'required|string',
            'latitude' => 'required|numeric',
            'longitude' => 'required|numeric',
            'delivery_radius' => 'nullable|numeric',
            'opening_time' => 'nullable|date_format:H:i',
            'closing_time' => 'nullable|date_format:H:i',
        ]);

        $restaurant = Restaurant::create([
            'user_id' => $request->user()->id,
            'name' => $request->name,
            'slug' => Str::slug($request->name),
            'description' => $request->description,
            'phone' => $request->phone,
            'email' => $request->email,
            'address' => $request->address,
            'latitude' => $request->latitude,
            'longitude' => $request->longitude,
            'delivery_radius' => $request->delivery_radius ?? 10,
            'minimum_order' => $request->minimum_order ?? 500,
            'delivery_fee' => $request->delivery_fee ?? 200,
            'opening_time' => $request->opening_time,
            'closing_time' => $request->closing_time,
            'status' => 'approved',
            'operational_status' => 'open',
        ]);

        return response()->json($restaurant, 201);
    }

    public function show($id)
    {
        $restaurant = Restaurant::findOrFail($id);

        $data = $restaurant->toArray();
        $data['is_open'] = $restaurant->operational_status === 'open';
        $data['delivery_time'] = $restaurant->estimated_delivery_time ?? 30;
        $data['tags'] = [];
        $data['delivery_enabled'] = $restaurant->delivery_enabled ?? true;
        $data['delivery_slots'] = $restaurant->delivery_slots;
        $data['opening_time'] = $restaurant->opening_time;
        $data['closing_time'] = $restaurant->closing_time;
        $data['operational_status'] = $restaurant->operational_status;

        return response()->json(['data' => $data]);
    }

    public function update(Request $request, $id)
    {
        $restaurant = Restaurant::findOrFail($id);
        
        $data = $request->only([
            'name', 'description', 'address', 'phone', 'email',
            'latitude', 'longitude', 'delivery_radius', 'minimum_order',
            'delivery_fee', 'opening_time', 'closing_time'
        ]);
        
        if (isset($data['name'])) {
            $data['slug'] = Str::slug($data['name']);
        }
        
        $restaurant->update($data);
        return response()->json($restaurant);
    }

    public function destroy($id)
    {
        $restaurant = Restaurant::findOrFail($id);
        $restaurant->delete();
        return response()->json(['message' => 'Restaurant deleted successfully']);
    }
}
