<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;

class UserController extends Controller
{
    public function profile()
    {
        return response()->json(auth()->user()->load('roles'));
    }

    public function updateProfile(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'nullable|string|max:20',
            'address' => 'nullable|string'
        ]);

        auth()->user()->update($validated);

        return response()->json(auth()->user());
    }

    public function changePassword(Request $request)
    {
        $validated = $request->validate([
            'current' => 'required',
            'new' => 'required|min:6',
            'confirm' => 'required|same:new'
        ]);

        if (!Hash::check($validated['current'], auth()->user()->password)) {
            return response()->json(['message' => 'Current password is incorrect'], 422);
        }

        auth()->user()->update([
            'password' => Hash::make($validated['new'])
        ]);

        return response()->json(['message' => 'Password changed successfully']);
    }

    public function stats()
    {
        $user = auth()->user();
        
        $totalOrders = $user->orders()->count();
        $walletBalance = $user->wallet->balance ?? 0;
        $loyaltyPoints = $user->loyalty_points ?? 0;
        $totalReviews = $user->reviews()->count();

        return response()->json([
            'totalOrders' => $totalOrders,
            'walletBalance' => $walletBalance,
            'loyaltyPoints' => $loyaltyPoints,
            'totalReviews' => $totalReviews
        ]);
    }

    public function uploadAvatar(Request $request)
    {
        $validated = $request->validate([
            'avatar' => 'required|image|max:2048'
        ]);

        $user = auth()->user();
        
        if ($request->hasFile('avatar')) {
            $path = $request->file('avatar')->store('avatars', 'public');
            $user->update(['avatar' => '/storage/' . $path]);
        }

        return response()->json(['avatar' => $user->avatar]);
    }
}
