<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;

class Order extends Model
{
    use SoftDeletes;

    protected $fillable = [
        'order_number', 'user_id', 'restaurant_id', 'status', 'delivery_address',
        'delivery_latitude', 'delivery_longitude', 'delivery_phone', 'delivery_instructions',
        'subtotal', 'delivery_fee', 'service_charge', 'tax', 'discount', 'total',
        'distance_km', 'estimated_delivery_time', 'payment_method', 'payment_status',
        'customer_notes', 'kitchen_notes', 'delivery_notes', 'cancellation_reason',
        'accepted_at', 'preparing_at', 'ready_at', 'picked_up_at', 'delivered_at', 'cancelled_at',
        'refund_amount', 'refund_reason', 'refund_status', 'refunded_by', 'refunded_at', 'cancelled_by'
    ];

    protected $casts = [
        'delivery_latitude' => 'decimal:7',
        'delivery_longitude' => 'decimal:7',
        'subtotal' => 'decimal:2',
        'delivery_fee' => 'decimal:2',
        'service_charge' => 'decimal:2',
        'tax' => 'decimal:2',
        'discount' => 'decimal:2',
        'total' => 'decimal:2',
        'distance_km' => 'decimal:2',
        'accepted_at' => 'datetime',
        'preparing_at' => 'datetime',
        'ready_at' => 'datetime',
        'picked_up_at' => 'datetime',
        'delivered_at' => 'datetime',
        'cancelled_at' => 'datetime',
        'refunded_at' => 'datetime',
        'refund_amount' => 'decimal:2',
    ];

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function restaurant(): BelongsTo
    {
        return $this->belongsTo(Restaurant::class);
    }

    public function items(): HasMany
    {
        return $this->hasMany(OrderItem::class);
    }

    public function delivery(): HasOne
    {
        return $this->hasOne(Delivery::class);
    }

    public function payment(): HasOne
    {
        return $this->hasOne(Payment::class);
    }

    public function reviews(): HasMany
    {
        return $this->hasMany(Review::class);
    }

    public function statusHistories(): HasMany
    {
        return $this->hasMany(OrderStatusHistory::class);
    }

    public function adminActions(): HasMany
    {
        return $this->hasMany(OrderAdminAction::class);
    }

    public function financial(): HasOne
    {
        return $this->hasOne(OrderFinancial::class);
    }

    public function flags(): HasMany
    {
        return $this->hasMany(OrderFlag::class);
    }

    public function auditTrails(): HasMany
    {
        return $this->hasMany(OrderAuditTrail::class);
    }

    public function financialSnapshot(): HasOne
    {
        return $this->hasOne(OrderFinancialSnapshot::class);
    }

    protected static function booted()
    {
        static::creating(function ($order) {
            if (empty($order->order_number)) {
                $order->order_number = self::generateOrderNumber($order->restaurant_id);
            }
        });
        
        static::created(function ($order) {
            $financialData = OrderFinancial::calculateFromOrder($order);
            $order->financial()->create(array_merge($financialData, ['order_id' => $order->id]));
            
            // Create financial snapshot
            \App\Services\FinancialSnapshotService::createSnapshot($order);
        });

        static::updating(function ($order) {
            // Status validation disabled - handled in OrderController
        });
    }
    
    public static function generateOrderNumber($restaurantId)
    {
        $restaurant = Restaurant::find($restaurantId);
        $restaurantCode = $restaurant ? strtoupper(substr($restaurant->name, 0, 3)) : 'ORD';
        $date = now()->format('ymd');
        $hour = now()->format('H');
        
        // Get count of orders for this restaurant today at this hour
        $count = self::where('restaurant_id', $restaurantId)
            ->whereDate('created_at', now()->toDateString())
            ->whereRaw('HOUR(created_at) = ?', [$hour])
            ->count() + 1;
        
        return $restaurantCode . $date . $hour . str_pad($count, 3, '0', STR_PAD_LEFT);
    }
}
