<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Database\Eloquent\Relations\MorphOne;

class Restaurant extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'user_id', 'name', 'slug', 'description', 'phone', 'email', 'address',
        'latitude', 'longitude', 'opening_time', 'closing_time', 'delivery_radius', 'minimum_order', 'delivery_fee',
        'service_charge_percentage', 'tax_percentage',
        'status', 'operational_status', 'logo', 'cover_image', 'images', 'rating',
        'total_reviews', 'is_featured', 'commission_rate', 'monthly_fee',
        'next_payment_date', 'last_payment_date', 'payment_status',
        'is_blocked', 'blocked_reason', 'blocked_by', 'blocked_at', 'blocked_until',
        'unblocked_by', 'unblocked_at', 'unblock_reason',
        'delivery_enabled', 'delivery_slots'
    ];

    protected $casts = [
        'images' => 'array',
        'delivery_slots' => 'array',
        'latitude' => 'decimal:7',
        'longitude' => 'decimal:7',
        'delivery_radius' => 'decimal:2',
        'minimum_order' => 'decimal:2',
        'delivery_fee' => 'decimal:2',
        'service_charge_percentage' => 'decimal:2',
        'tax_percentage' => 'decimal:2',
        'rating' => 'decimal:2',
        'is_featured' => 'boolean',
        'delivery_enabled' => 'boolean',
        'is_blocked' => 'boolean',
        'blocked_at' => 'datetime',
        'blocked_until' => 'datetime',
        'unblocked_at' => 'datetime',
    ];

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function owner(): BelongsTo
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    public function products(): BelongsToMany
    {
        return $this->belongsToMany(Product::class, 'restaurant_products')
            ->withPivot(
                'price', 'offer_price', 'is_available', 'is_spicy', 'is_vegetarian', 'is_featured', 
                'preparation_time', 'stock_quantity', 'discount_percentage', 'special_notes', 
                'custom_image', 'available_from', 'available_until', 'min_order_quantity',
                'special_start_date', 'special_end_date'
            )
            ->withTimestamps();
    }

    public function productsNew(): HasMany
    {
        return $this->hasMany(ProductNew::class);
    }

    public function orders(): HasMany
    {
        return $this->hasMany(Order::class);
    }

    public function reviews(): MorphMany
    {
        return $this->morphMany(Review::class, 'reviewable');
    }

    public function commissionRule(): MorphOne
    {
        return $this->morphOne(CommissionRule::class, 'commissionable');
    }

    public function restaurantCategories(): HasMany
    {
        return $this->hasMany(RestaurantCategory::class);
    }

    // Remove the old categories relationship that uses pivot table
    // public function categories(): BelongsToMany
    // {
    //     return $this->belongsToMany(Category::class, 'restaurant_categories')
    //                 ->withPivot('is_active', 'sort_order')
    //                 ->withTimestamps();
    // }

    public function payments(): HasMany
    {
        return $this->hasMany(RestaurantPayment::class)->orderBy('payment_date', 'desc');
    }

    public function blockedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'blocked_by');
    }

    public function unblockedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'unblocked_by');
    }

    public function scopeNotBlocked($query)
    {
        return $query->where(function($q) {
            $q->where('is_blocked', false)
              ->orWhere(function($q2) {
                  $q2->where('is_blocked', true)
                     ->where('blocked_until', '<=', now());
              });
        });
    }

    public function scopeBlocked($query)
    {
        return $query->where('is_blocked', true)
                    ->where(function($q) {
                        $q->whereNull('blocked_until')
                          ->orWhere('blocked_until', '>', now());
                    });
    }
}
