<?php

namespace App\Repositories;

use App\Models\CommissionRule;
use Illuminate\Database\Eloquent\Builder;

class CommissionRuleRepository
{
    public function create(array $data)
    {
        return CommissionRule::create($data);
    }

    public function update(CommissionRule $commissionRule, array $data)
    {
        $commissionRule->update($data);
        return $commissionRule->fresh();
    }

    public function delete(CommissionRule $commissionRule)
    {
        return $commissionRule->delete();
    }

    public function findById($id)
    {
        return CommissionRule::with(['commissionable'])->findOrFail($id);
    }

    public function findByCommissionable($commissionableId, $commissionableType)
    {
        return CommissionRule::where('commissionable_id', $commissionableId)
            ->where('commissionable_type', $commissionableType)
            ->first();
    }

    public function updateOrCreate(array $attributes, array $values)
    {
        return CommissionRule::updateOrCreate($attributes, $values);
    }

    public function search(array $filters = [])
    {
        $query = CommissionRule::with(['commissionable']);

        // Apply filters
        if (isset($filters['type'])) {
            $query->where('type', $filters['type']);
        }

        if (isset($filters['status'])) {
            $query->where('status', $filters['status']);
        }

        if (isset($filters['commissionable_type'])) {
            $query->where('commissionable_type', $filters['commissionable_type']);
        }

        if (isset($filters['commissionable_id'])) {
            $query->where('commissionable_id', $filters['commissionable_id']);
        }

        if (isset($filters['search'])) {
            $search = $filters['search'];
            $query->where(function (Builder $q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%");
            });
        }

        // Apply sorting
        $sortBy = $filters['sort_by'] ?? 'created_at';
        $sortOrder = $filters['sort_order'] ?? 'desc';
        $query->orderBy($sortBy, $sortOrder);

        // Apply pagination
        $perPage = $filters['per_page'] ?? 15;
        return $query->paginate($perPage);
    }

    public function getActiveCommissionRules()
    {
        return CommissionRule::where('status', 'active')
            ->with(['commissionable'])
            ->orderBy('name')
            ->get();
    }

    public function getCommissionRulesByType(string $type)
    {
        return CommissionRule::where('type', $type)
            ->where('status', 'active')
            ->with(['commissionable'])
            ->orderBy('name')
            ->get();
    }

    public function getCommissionRulesByCommissionableType(string $commissionableType)
    {
        return CommissionRule::where('commissionable_type', $commissionableType)
            ->where('status', 'active')
            ->with(['commissionable'])
            ->orderBy('name')
            ->get();
    }
}