<?php

namespace App\Services;

use App\Models\DeliveryBoy;
use App\Models\DeliveryBoyEarning;
use App\Models\Order;
use App\Models\Delivery;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class DeliveryBoyEarningService
{
    /**
     * Calculate and record earning for a delivery
     */
    public function recordDeliveryEarning(Delivery $delivery, Order $order): DeliveryBoyEarning
    {
        $deliveryBoy = $delivery->deliveryBoy;
        $deliveryFee = $order->delivery_fee ?? 0;
        
        // Calculate platform commission
        $platformCommission = ($deliveryFee * $deliveryBoy->commission_rate) / 100;
        $deliveryBoyEarning = $deliveryFee - $platformCommission;

        // Record earning
        $earning = DeliveryBoyEarning::create([
            'delivery_boy_id' => $deliveryBoy->id,
            'order_id' => $order->id,
            'delivery_id' => $delivery->id,
            'earning_type' => DeliveryBoyEarning::TYPE_DELIVERY_FEE,
            'amount' => $deliveryBoyEarning,
            'commission_rate' => $deliveryBoy->commission_rate,
            'description' => "Delivery fee for order #{$order->order_number}",
            'status' => DeliveryBoyEarning::STATUS_PENDING,
        ]);

        // Update delivery boy wallet
        $deliveryBoy->increment('wallet_balance', $deliveryBoyEarning);
        $deliveryBoy->increment('total_deliveries');
        $deliveryBoy->increment('successful_deliveries');

        return $earning;
    }

    /**
     * Add bonus to delivery boy
     */
    public function addBonus(DeliveryBoy $deliveryBoy, float $amount, string $description): DeliveryBoyEarning
    {
        $earning = DeliveryBoyEarning::create([
            'delivery_boy_id' => $deliveryBoy->id,
            'earning_type' => DeliveryBoyEarning::TYPE_BONUS,
            'amount' => $amount,
            'description' => $description,
            'status' => DeliveryBoyEarning::STATUS_PENDING,
            'recorded_by' => auth()->id(),
        ]);

        $deliveryBoy->increment('wallet_balance', $amount);

        return $earning;
    }

    /**
     * Add tip to delivery boy
     */
    public function addTip(DeliveryBoy $deliveryBoy, Order $order, float $amount): DeliveryBoyEarning
    {
        $earning = DeliveryBoyEarning::create([
            'delivery_boy_id' => $deliveryBoy->id,
            'order_id' => $order->id,
            'earning_type' => DeliveryBoyEarning::TYPE_TIP,
            'amount' => $amount,
            'description' => "Tip from order #{$order->order_number}",
            'status' => DeliveryBoyEarning::STATUS_PENDING,
        ]);

        $deliveryBoy->increment('wallet_balance', $amount);

        return $earning;
    }

    /**
     * Apply penalty to delivery boy
     */
    public function applyPenalty(DeliveryBoy $deliveryBoy, float $amount, string $reason): DeliveryBoyEarning
    {
        $earning = DeliveryBoyEarning::create([
            'delivery_boy_id' => $deliveryBoy->id,
            'earning_type' => DeliveryBoyEarning::TYPE_PENALTY,
            'amount' => -abs($amount),
            'description' => $reason,
            'status' => DeliveryBoyEarning::STATUS_PENDING,
            'recorded_by' => auth()->id(),
        ]);

        $deliveryBoy->decrement('wallet_balance', abs($amount));

        return $earning;
    }

    /**
     * Get pending earnings for delivery boy
     */
    public function getPendingEarnings(DeliveryBoy $deliveryBoy): float
    {
        return DeliveryBoyEarning::where('delivery_boy_id', $deliveryBoy->id)
            ->where('status', DeliveryBoyEarning::STATUS_PENDING)
            ->sum('amount');
    }

    /**
     * Get monthly earnings summary
     */
    public function getMonthlyEarnings(DeliveryBoy $deliveryBoy, ?Carbon $month = null): array
    {
        $month = $month ?? Carbon::now();
        $startDate = $month->copy()->startOfMonth();
        $endDate = $month->copy()->endOfMonth();

        $earnings = DeliveryBoyEarning::where('delivery_boy_id', $deliveryBoy->id)
            ->whereBetween('created_at', [$startDate, $endDate])
            ->selectRaw('
                earning_type,
                SUM(amount) as total,
                COUNT(*) as count
            ')
            ->groupBy('earning_type')
            ->get();

        $totalEarnings = $earnings->sum('total');
        $totalDeliveries = DeliveryBoyEarning::where('delivery_boy_id', $deliveryBoy->id)
            ->where('earning_type', DeliveryBoyEarning::TYPE_DELIVERY_FEE)
            ->whereBetween('created_at', [$startDate, $endDate])
            ->count();

        return [
            'month' => $month->format('Y-m'),
            'total_earnings' => $totalEarnings,
            'total_deliveries' => $totalDeliveries,
            'breakdown' => $earnings,
            'pending' => $this->getPendingEarnings($deliveryBoy),
            'paid' => DeliveryBoyEarning::where('delivery_boy_id', $deliveryBoy->id)
                ->where('status', DeliveryBoyEarning::STATUS_PAID)
                ->whereBetween('created_at', [$startDate, $endDate])
                ->sum('amount'),
        ];
    }

    /**
     * Process payout for delivery boy
     */
    public function processPayout(
        DeliveryBoy $deliveryBoy,
        float $amount,
        string $paymentMethod,
        ?string $transactionId = null
    ): bool {
        if ($amount > $deliveryBoy->wallet_balance) {
            throw new \Exception('Insufficient wallet balance');
        }

        DB::beginTransaction();
        try {
            // Mark pending earnings as paid
            $earnings = DeliveryBoyEarning::where('delivery_boy_id', $deliveryBoy->id)
                ->where('status', DeliveryBoyEarning::STATUS_PENDING)
                ->where('amount', '>', 0)
                ->get();

            $totalPaid = 0;
            foreach ($earnings as $earning) {
                if ($totalPaid + $earning->amount <= $amount) {
                    $earning->update([
                        'status' => DeliveryBoyEarning::STATUS_PAID,
                        'payment_date' => now(),
                        'payment_method' => $paymentMethod,
                        'transaction_id' => $transactionId,
                        'recorded_by' => auth()->id(),
                    ]);
                    $totalPaid += $earning->amount;
                }
            }

            // Deduct from wallet
            $deliveryBoy->decrement('wallet_balance', $totalPaid);

            DB::commit();
            return true;
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    /**
     * Get earnings report for date range
     */
    public function getEarningsReport(DeliveryBoy $deliveryBoy, Carbon $startDate, Carbon $endDate): array
    {
        $earnings = DeliveryBoyEarning::where('delivery_boy_id', $deliveryBoy->id)
            ->whereBetween('created_at', [$startDate, $endDate])
            ->get();

        return [
            'period' => [
                'start' => $startDate->format('Y-m-d'),
                'end' => $endDate->format('Y-m-d'),
            ],
            'total_earnings' => $earnings->sum('amount'),
            'total_deliveries' => $earnings->where('earning_type', DeliveryBoyEarning::TYPE_DELIVERY_FEE)->count(),
            'delivery_fees' => $earnings->where('earning_type', DeliveryBoyEarning::TYPE_DELIVERY_FEE)->sum('amount'),
            'bonuses' => $earnings->where('earning_type', DeliveryBoyEarning::TYPE_BONUS)->sum('amount'),
            'tips' => $earnings->where('earning_type', DeliveryBoyEarning::TYPE_TIP)->sum('amount'),
            'penalties' => $earnings->where('earning_type', DeliveryBoyEarning::TYPE_PENALTY)->sum('amount'),
            'pending' => $earnings->where('status', DeliveryBoyEarning::STATUS_PENDING)->sum('amount'),
            'paid' => $earnings->where('status', DeliveryBoyEarning::STATUS_PAID)->sum('amount'),
            'average_per_delivery' => $earnings->where('earning_type', DeliveryBoyEarning::TYPE_DELIVERY_FEE)->avg('amount'),
        ];
    }
}
