<?php

namespace App\Services;

use App\Models\Order;
use App\Models\OrderFinancialSnapshot;
use App\Exceptions\ImmutableFinancialDataException;

class FinancialSnapshotService
{
    public static function createSnapshot(Order $order): OrderFinancialSnapshot
    {
        // Check if snapshot already exists
        if ($order->financialSnapshot) {
            return $order->financialSnapshot;
        }

        return OrderFinancialSnapshot::createFromOrder($order);
    }

    public static function lockSnapshot(Order $order): void
    {
        $snapshot = $order->financialSnapshot;
        
        if (!$snapshot) {
            throw new \Exception('No financial snapshot exists for this order');
        }

        if ($order->status !== 'delivered') {
            throw new \Exception('Can only lock financial snapshot after delivery');
        }

        $snapshot->makeImmutable();
        
        OrderAuditService::logManualOverride(
            $order, 
            ['financial_snapshot_locked' => true], 
            'Financial snapshot locked after delivery'
        );
    }

    public static function processRefund(Order $order, float $refundAmount): void
    {
        $snapshot = $order->financialSnapshot;
        
        if (!$snapshot) {
            throw new \Exception('No financial snapshot exists for this order');
        }

        if ($snapshot->is_immutable) {
            throw new ImmutableFinancialDataException('Cannot process refund on immutable financial snapshot');
        }

        $snapshot->updateRefund($refundAmount);
        
        // Recalculate earnings after refund
        $newRestaurantEarning = $snapshot->restaurant_earning - $refundAmount;
        $snapshot->update(['restaurant_earning' => max(0, $newRestaurantEarning)]);
    }

    public static function updateSettlement(Order $order, string $status): void
    {
        $snapshot = $order->financialSnapshot;
        
        if (!$snapshot) {
            throw new \Exception('No financial snapshot exists for this order');
        }

        $snapshot->updateSettlementStatus($status);
    }

    public static function getFinancialSummary(array $orderIds = null): array
    {
        $query = OrderFinancialSnapshot::query();
        
        if ($orderIds) {
            $query->whereIn('order_id', $orderIds);
        }

        $snapshots = $query->get();

        return [
            'total_customer_payments' => $snapshots->sum('customer_payment'),
            'total_platform_commission' => $snapshots->sum('platform_commission'),
            'total_restaurant_earnings' => $snapshots->sum('restaurant_earning'),
            'total_delivery_earnings' => $snapshots->sum('delivery_boy_earning'),
            'total_refunds' => $snapshots->sum('refund_amount'),
            'settlement_breakdown' => $snapshots->groupBy('settlement_status')->map->count(),
            'immutable_count' => $snapshots->where('is_immutable', true)->count(),
            'mutable_count' => $snapshots->where('is_immutable', false)->count()
        ];
    }

    public static function validateFinancialIntegrity(Order $order): array
    {
        $snapshot = $order->financialSnapshot;
        
        if (!$snapshot) {
            return ['valid' => false, 'message' => 'No financial snapshot exists'];
        }

        $expectedTotal = $snapshot->restaurant_earning + 
                        $snapshot->platform_commission + 
                        $snapshot->delivery_boy_earning;

        $actualTotal = $snapshot->customer_payment - $snapshot->refund_amount;

        $isValid = abs($expectedTotal - $actualTotal) < 0.01; // Allow for rounding

        return [
            'valid' => $isValid,
            'expected_total' => $expectedTotal,
            'actual_total' => $actualTotal,
            'difference' => $expectedTotal - $actualTotal,
            'is_immutable' => $snapshot->is_immutable
        ];
    }
}