<?php

namespace App\Services;

use App\Models\Order;
use App\Models\DeliveryBoy;
use App\Repositories\OrderRepository;
use App\Repositories\DeliveryBoyRepository;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Notification;
use App\Notifications\OrderStatusUpdated;
use App\Notifications\OrderAssignedToDeliveryBoy;

class OrderService
{
    protected $orderRepository;
    protected $deliveryBoyRepository;

    public function __construct(
        OrderRepository $orderRepository,
        DeliveryBoyRepository $deliveryBoyRepository
    ) {
        $this->orderRepository = $orderRepository;
        $this->deliveryBoyRepository = $deliveryBoyRepository;
    }

    public function updateOrderStatus(Order $order, string $status, array $data = [])
    {
        return DB::transaction(function () use ($order, $status, $data) {
            $previousStatus = $order->status;
            
            // Update order status
            $order->update([
                'status' => $status,
                'status_updated_at' => now()
            ]);

            // Add status history
            $order->statusHistory()->create([
                'status' => $status,
                'previous_status' => $previousStatus,
                'note' => $data['note'] ?? null,
                'updated_by' => request()->user()->id ?? null
            ]);

            // Handle specific status updates
            switch ($status) {
                case 'confirmed':
                    $this->handleOrderConfirmed($order);
                    break;
                case 'preparing':
                    $this->handleOrderPreparing($order);
                    break;
                case 'ready_for_pickup':
                    $this->handleOrderReadyForPickup($order);
                    break;
                case 'out_for_delivery':
                    $this->handleOrderOutForDelivery($order, $data);
                    break;
                case 'delivered':
                    $this->handleOrderDelivered($order, $data);
                    break;
                case 'cancelled':
                    $this->handleOrderCancelled($order, $data);
                    break;
            }

            // Notify relevant parties
            $this->notifyOrderStatusChange($order, $previousStatus, $status);

            return $order->fresh();
        });
    }

    public function assignDeliveryBoy(Order $order, DeliveryBoy $deliveryBoy)
    {
        return DB::transaction(function () use ($order, $deliveryBoy) {
            // Update order
            $order->update([
                'delivery_boy_id' => $deliveryBoy->id,
                'assigned_at' => now(),
                'assigned_by' => request()->user()->id ?? null
            ]);

            // Update delivery boy status
            $deliveryBoy->update(['is_available' => false]);

            // Add status history
            $order->statusHistory()->create([
                'status' => 'assigned',
                'previous_status' => $order->status,
                'note' => "Assigned to {$deliveryBoy->name}",
                'updated_by' => request()->user()->id ?? null
            ]);

            // Notify delivery boy
            $deliveryBoy->user->notify(new OrderAssignedToDeliveryBoy($order));

            return $order->fresh();
        });
    }

    public function calculateCommission(Order $order)
    {
        $restaurant = $order->restaurant;
        $commissionRule = $restaurant->commissionRule;

        if (!$commissionRule || !$commissionRule->is_active) {
            return 0;
        }

        $commissionAmount = 0;

        if ($commissionRule->type === 'percentage') {
            $commissionAmount = ($order->subtotal * $commissionRule->rate) / 100;
        } else {
            $commissionAmount = $commissionRule->rate;
        }

        return round($commissionAmount, 2);
    }

    public function getOrderStats(array $filters = [])
    {
        $query = Order::query();

        // Apply filters
        if (isset($filters['restaurant_id'])) {
            $query->where('restaurant_id', $filters['restaurant_id']);
        }

        if (isset($filters['status'])) {
            $query->where('status', $filters['status']);
        }

        if (isset($filters['date_from'])) {
            $query->whereDate('created_at', '>=', $filters['date_from']);
        }

        if (isset($filters['date_to'])) {
            $query->whereDate('created_at', '<=', $filters['date_to']);
        }

        return [
            'total_orders' => $query->count(),
            'total_revenue' => $query->where('status', 'delivered')->sum('total_amount'),
            'average_order_value' => $query->where('status', 'delivered')->avg('total_amount') ?? 0,
            'cancelled_orders' => (clone $query)->where('status', 'cancelled')->count(),
            'cancelled_revenue' => (clone $query)->where('status', 'cancelled')->sum('total_amount'),
            'status_distribution' => $query->select('status', DB::raw('count(*) as count'))
                ->groupBy('status')
                ->pluck('count', 'status')
                ->toArray()
        ];
    }

    public function searchOrders(array $filters = [])
    {
        return $this->orderRepository->search($filters);
    }

    private function handleOrderConfirmed(Order $order)
    {
        // Notify restaurant
        $order->restaurant->owner->notify(new OrderStatusUpdated($order, 'confirmed'));
        
        // Update preparation time
        $order->update([
            'confirmed_at' => now(),
            'estimated_preparation_time' => now()->addMinutes($order->restaurant->average_preparation_time ?? 30)
        ]);
    }

    private function handleOrderPreparing(Order $order)
    {
        $order->update([
            'preparation_started_at' => now()
        ]);
    }

    private function handleOrderReadyForPickup(Order $order)
    {
        $order->update([
            'ready_for_pickup_at' => now()
        ]);

        // Auto-assign delivery boy if enabled
        if (config('delivery.auto_assign_delivery_boy')) {
            $this->autoAssignDeliveryBoy($order);
        }
    }

    private function handleOrderOutForDelivery(Order $order, array $data)
    {
        $order->update([
            'out_for_delivery_at' => now(),
            'estimated_delivery_time' => $data['estimated_delivery_time'] ?? now()->addMinutes(30)
        ]);
    }

    private function handleOrderDelivered(Order $order, array $data)
    {
        $order->update([
            'delivered_at' => now(),
            'actual_delivery_time' => $data['actual_delivery_time'] ?? now()
        ]);

        // Calculate and record commission
        $commissionAmount = $this->calculateCommission($order);
        $order->commission()->create([
            'amount' => $commissionAmount,
            'type' => 'restaurant_commission',
            'calculated_at' => now()
        ]);

        // Mark delivery boy as available
        if ($order->deliveryBoy) {
            $order->deliveryBoy->update(['is_available' => true]);
        }
    }

    private function handleOrderCancelled(Order $order, array $data)
    {
        $order->update([
            'cancelled_at' => now(),
            'cancellation_reason' => $data['reason'] ?? null,
            'cancelled_by' => request()->user()->id ?? null
        ]);

        // Refund payment if already paid
        if ($order->payment_status === 'paid') {
            $this->processRefund($order);
        }

        // Mark delivery boy as available
        if ($order->deliveryBoy) {
            $order->deliveryBoy->update(['is_available' => true]);
        }
    }

    private function autoAssignDeliveryBoy(Order $order)
    {
        // Find available delivery boys near the restaurant
        $deliveryBoys = $this->deliveryBoyRepository->getAvailableDeliveryBoys(
            $order->restaurant->latitude,
            $order->restaurant->longitude,
            config('delivery.delivery_boy_radius', 5)
        );

        if ($deliveryBoys->isNotEmpty()) {
            // Assign to the nearest delivery boy
            $deliveryBoy = $deliveryBoys->first();
            $this->assignDeliveryBoy($order, $deliveryBoy);
        }
    }

    private function processRefund(Order $order)
    {
        // Implement refund logic based on payment method
        // This would integrate with your payment gateway
        $order->refund()->create([
            'amount' => $order->total_amount,
            'status' => 'pending',
            'processed_at' => now(),
            'processed_by' => request()->user()->id ?? null
        ]);
    }

    private function notifyOrderStatusChange(Order $order, string $previousStatus, string $newStatus)
    {
        // Notify customer
        $order->user->notify(new OrderStatusUpdated($order, $newStatus));

        // Notify restaurant for certain status changes
        if (in_array($newStatus, ['confirmed', 'cancelled', 'delivered'])) {
            $order->restaurant->owner->notify(new OrderStatusUpdated($order, $newStatus));
        }

        // Notify delivery boy if assigned
        if ($order->deliveryBoy && in_array($newStatus, ['ready_for_pickup', 'out_for_delivery', 'delivered'])) {
            $order->deliveryBoy->user->notify(new OrderStatusUpdated($order, $newStatus));
        }
    }
}