<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\{User, Restaurant, Product, Order, Category, DeliveryBoy, Cart, OrderItem, Payment, Wallet, Review, PromoCode};
use Carbon\Carbon;

class DemoDataSeeder extends Seeder
{
    public function run(): void
    {
        // Create categories
        $categories = [
            ['name' => 'Pizza', 'slug' => 'pizza', 'description' => 'Delicious pizzas', 'is_active' => true, 'sort_order' => 1],
            ['name' => 'Burgers', 'slug' => 'burgers', 'description' => 'Juicy burgers', 'is_active' => true, 'sort_order' => 2],
            ['name' => 'Pasta', 'slug' => 'pasta', 'description' => 'Italian pasta', 'is_active' => true, 'sort_order' => 3],
            ['name' => 'Chinese', 'slug' => 'chinese', 'description' => 'Chinese cuisine', 'is_active' => true, 'sort_order' => 4],
            ['name' => 'Desserts', 'slug' => 'desserts', 'description' => 'Sweet desserts', 'is_active' => true, 'sort_order' => 5],
        ];

        foreach ($categories as $cat) {
            Category::firstOrCreate(['slug' => $cat['slug']], $cat);
        }

        // Get users
        $manager = User::where('email', 'manager@fooddelivery.com')->first();
        $customer = User::where('email', 'customer@fooddelivery.com')->first();
        $deliveryUser = User::where('email', 'delivery@fooddelivery.com')->first();

        if (!$manager) $manager = User::first();
        if (!$customer) $customer = User::skip(1)->first();
        if (!$deliveryUser) $deliveryUser = User::skip(2)->first();

        // Create restaurants
        $restaurants = [
            [
                'user_id' => $manager->id,
                'name' => 'Pizza Palace',
                'description' => 'Best pizza in town with authentic Italian flavors',
                'email' => 'pizza@example.com',
                'phone' => '0771234567',
                'address' => '123 Main Street, Colombo 03',
                'latitude' => 6.9271,
                'longitude' => 79.8612,
                'status' => 'approved',
                'operational_status' => 'open',
                'rating' => 4.5,
            ],
            [
                'user_id' => $manager->id,
                'name' => 'Burger House',
                'description' => 'Delicious burgers and fries made fresh daily',
                'email' => 'burger@example.com',
                'phone' => '0771234568',
                'address' => '456 Food Street, Colombo 07',
                'latitude' => 6.9271,
                'longitude' => 79.8612,
                'status' => 'approved',
                'operational_status' => 'open',
                'rating' => 4.2,
            ],
        ];

        foreach ($restaurants as $rest) {
            Restaurant::firstOrCreate(['name' => $rest['name']], $rest);
        }

        // Create products
        $pizzaRestaurant = Restaurant::where('name', 'Pizza Palace')->first();
        $burgerRestaurant = Restaurant::where('name', 'Burger House')->first();

        $products = [
            [
                'restaurant_id' => $pizzaRestaurant->id,
                'category_id' => 1,
                'name' => 'Margherita Pizza',
                'slug' => 'margherita-pizza',
                'description' => 'Classic pizza with tomato sauce, mozzarella, and fresh basil',
                'price' => 1200.00,
                'image' => 'pizza-margherita.jpg',
                'is_available' => true,
                'is_active' => true,
                'is_approved' => true,
                'stock_quantity' => 50,
                'rating' => 4.5,
                'total_reviews' => 25,
                'total_orders' => 150,
            ],
            [
                'restaurant_id' => $pizzaRestaurant->id,
                'category_id' => 1,
                'name' => 'Pepperoni Pizza',
                'slug' => 'pepperoni-pizza',
                'description' => 'Spicy pepperoni with mozzarella cheese and tomato sauce',
                'price' => 1500.00,
                'image' => 'pizza-pepperoni.jpg',
                'is_available' => true,
                'is_active' => true,
                'is_approved' => true,
                'stock_quantity' => 40,
                'rating' => 4.3,
                'total_reviews' => 18,
                'total_orders' => 120,
            ],
            [
                'restaurant_id' => $burgerRestaurant->id,
                'category_id' => 2,
                'name' => 'Cheese Burger',
                'slug' => 'cheese-burger',
                'description' => 'Juicy beef patty with cheese, lettuce, tomato, and special sauce',
                'price' => 800.00,
                'image' => 'burger-cheese.jpg',
                'is_available' => true,
                'is_active' => true,
                'is_approved' => true,
                'stock_quantity' => 60,
                'rating' => 4.2,
                'total_reviews' => 32,
                'total_orders' => 200,
            ],
        ];

        foreach ($products as $prod) {
            Product::firstOrCreate(['slug' => $prod['slug']], $prod);
        }

        // Create delivery boy
        if ($deliveryUser) {
            DeliveryBoy::firstOrCreate(
                ['user_id' => $deliveryUser->id],
                [
                    'user_id' => $deliveryUser->id,
                    'full_name' => $deliveryUser->name,
                    'phone' => '0771234567',
                    'email' => $deliveryUser->email,
                    'address' => 'Delivery Boy Address, Colombo',
                    'city' => 'Colombo',
                    'postal_code' => '00300',
                    'status' => 'online',
                    'verification_status' => 'verified',
                    'national_id' => '123456789V',
                    'date_of_birth' => '1990-01-01',
                    'emergency_contact_name' => 'Emergency Contact',
                    'emergency_contact_phone' => '0771234568',
                    'rating' => 4.8,
                    'total_ratings' => 45,
                    'total_deliveries' => 150,
                    'successful_deliveries' => 145,
                    'commission_rate' => 15.00,
                    'wallet_balance' => 25000.00,
                    'is_available' => true,
                    'joined_date' => Carbon::now()->subMonths(6),
                    'last_active_at' => Carbon::now(),
                ]
            );
        }

        // Create sample orders
        if ($customer && $pizzaRestaurant) {
            for ($i = 1; $i <= 20; $i++) {
                $restaurant = Restaurant::inRandomOrder()->first();
                $orderDate = Carbon::now()->subDays(rand(0, 30));
                $subtotal = rand(800, 3000);
                $deliveryFee = 150;
                $serviceCharge = round($subtotal * 0.1);
                $tax = round($subtotal * 0.08);
                $total = $subtotal + $deliveryFee + $serviceCharge + $tax;
                
                $order = Order::create([
                    'order_number' => 'FD' . str_pad($i, 6, '0', STR_PAD_LEFT),
                    'user_id' => $customer->id,
                    'restaurant_id' => $restaurant->id,
                    'status' => ['pending', 'accepted', 'preparing', 'ready', 'delivered'][rand(0, 4)],
                    'delivery_address' => '123 Customer Street, Colombo 03',
                    'delivery_latitude' => 6.9271,
                    'delivery_longitude' => 79.8612,
                    'delivery_phone' => '0771234567',
                    'subtotal' => $subtotal,
                    'delivery_fee' => $deliveryFee,
                    'service_charge' => $serviceCharge,
                    'tax' => $tax,
                    'total' => $total,
                    'distance_km' => rand(2, 15),
                    'estimated_delivery_time' => rand(20, 60),
                    'payment_method' => ['cash', 'card', 'wallet'][rand(0, 2)],
                    'payment_status' => 'paid',
                    'customer_notes' => 'Please ring the bell',
                    'created_at' => $orderDate,
                    'updated_at' => $orderDate,
                ]);

                // Create order items
                $products = Product::where('restaurant_id', $restaurant->id)->take(rand(1, 3))->get();
                foreach ($products as $product) {
                    $qty = rand(1, 3);
                    OrderItem::create([
                        'order_id' => $order->id,
                        'product_id' => $product->id,
                        'product_name' => $product->name,
                        'quantity' => $qty,
                        'unit_price' => $product->price,
                        'total_price' => $product->price * $qty,
                        'special_notes' => 'No onions please',
                    ]);
                }
            }
        }

        echo "✅ Demo data created successfully!\n";
        echo "📊 Created:\n";
        echo "   - " . count($categories) . " categories\n";
        echo "   - " . count($restaurants) . " restaurants\n";
        echo "   - " . count($products) . " products\n";
        echo "   - 1 delivery boy\n";
        echo "   - 20 sample orders with items\n";
    }
}