<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\Restaurant;
use App\Models\User;
use App\Models\Product;

class MassOrderSeeder extends Seeder
{
    public function run()
    {
        $this->command->info('Creating 100 orders with 5-6 orders per restaurant...');
        
        $restaurants = Restaurant::limit(20)->get();
        $customers = User::role('customer')->limit(10)->get();
        $statuses = ['pending', 'accepted', 'preparing', 'ready_for_pickup', 'picked_up', 'delivered'];
        
        // Create fake products for restaurants if none exist
        foreach ($restaurants as $restaurant) {
            if (Product::where('restaurant_id', $restaurant->id)->count() == 0) {
                for ($p = 1; $p <= 5; $p++) {
                    Product::create([
                        'restaurant_id' => $restaurant->id,
                        'category_id' => 1,
                        'name' => "Product {$p} - {$restaurant->name}",
                        'slug' => "product-{$p}-{$restaurant->id}",
                        'description' => "Delicious food item {$p}",
                        'price' => rand(500, 2000),
                        'image' => 'default.jpg',
                        'is_available' => 1,
                    ]);
                }
            }
        }
        
        $orderCount = 0;
        
        foreach ($restaurants as $restaurant) {
            $products = Product::where('restaurant_id', $restaurant->id)->limit(5)->get();
            
            // Create 5-6 orders per restaurant
            $ordersPerRestaurant = rand(5, 6);
            
            for ($i = 0; $i < $ordersPerRestaurant && $orderCount < 100; $i++) {
                $customer = $customers->random();
                $orderNumber = 'FD' . str_pad(100000 + $orderCount, 6, '0', STR_PAD_LEFT);
                
                $subtotal = 0;
                $orderItems = [];
                
                // Add 2-4 items per order
                $itemCount = rand(2, 4);
                for ($j = 0; $j < $itemCount; $j++) {
                    $product = $products->random();
                    $quantity = rand(1, 3);
                    $unitPrice = rand(500, 2000);
                    $totalPrice = $unitPrice * $quantity;
                    $subtotal += $totalPrice;
                    
                    $orderItems[] = [
                        'product_id' => $product->id,
                        'product_name' => $product->name,
                        'quantity' => $quantity,
                        'unit_price' => $unitPrice,
                        'total_price' => $totalPrice,
                        'special_notes' => rand(0, 1) ? 'Extra spicy' : null,
                    ];
                }
                
                $deliveryFee = rand(100, 300);
                $serviceCharge = $subtotal * 0.1;
                $tax = $subtotal * 0.08;
                $total = $subtotal + $deliveryFee + $serviceCharge + $tax;
                
                $order = Order::create([
                    'order_number' => $orderNumber,
                    'user_id' => $customer->id,
                    'restaurant_id' => $restaurant->id,
                    'status' => $statuses[array_rand($statuses)],
                    'delivery_address' => $customer->address ?? 'Test Address, Colombo',
                    'delivery_latitude' => 6.9271 + (rand(-50, 50) / 1000),
                    'delivery_longitude' => 79.8612 + (rand(-50, 50) / 1000),
                    'delivery_phone' => $customer->phone,
                    'subtotal' => $subtotal,
                    'delivery_fee' => $deliveryFee,
                    'service_charge' => $serviceCharge,
                    'tax' => $tax,
                    'total' => $total,
                    'distance_km' => rand(1, 10),
                    'estimated_delivery_time' => rand(20, 60),
                    'payment_method' => rand(0, 1) ? 'cash' : 'card',
                    'payment_status' => 'paid',
                    'customer_notes' => 'Test order ' . ($orderCount + 1),
                ]);
                
                // Create order items
                foreach ($orderItems as $item) {
                    OrderItem::create(array_merge($item, ['order_id' => $order->id]));
                }
                
                $orderCount++;
            }
        }
        
        $this->command->info("Created {$orderCount} orders successfully!");
    }
}