<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\User;
use App\Models\Restaurant;
use App\Models\RestaurantCategory;
use App\Models\Category;
use App\Models\Product;
use Illuminate\Support\Facades\DB;

class RestaurantMenuSeeder extends Seeder
{
    public function run()
    {
        // Get or create manager user
        $manager = User::where('email', 'manager@test.com')->first();
        
        if (!$manager) {
            $manager = User::create([
                'name' => 'Test Manager',
                'email' => 'manager@test.com',
                'password' => bcrypt('password'),
                'phone' => '+94771234567',
                'address' => 'Colombo 07'
            ]);
            $manager->assignRole('manager');
        }

        // Get or create restaurant
        $restaurant = Restaurant::firstOrCreate(
            ['user_id' => $manager->id],
            [
                'name' => 'Spice Garden Restaurant',
                'description' => 'Authentic Sri Lankan cuisine with a modern twist',
                'address' => 'Colombo 07',
                'phone' => '0112345678',
                'latitude' => 6.9271,
                'longitude' => 79.8612,
                'status' => 'approved'
            ]
        );

        // Create restaurant categories
        $categories = [
            ['name' => 'Main Dishes', 'type' => 'main', 'sort_order' => 1],
            ['name' => 'Rice & Noodles', 'type' => 'main', 'sort_order' => 2],
            ['name' => 'Kottu', 'type' => 'main', 'sort_order' => 3],
            ['name' => 'Short Eats', 'type' => 'snack', 'sort_order' => 4],
            ['name' => 'Drinks', 'type' => 'drink', 'sort_order' => 5],
            ['name' => 'Desserts', 'type' => 'dessert', 'sort_order' => 6],
        ];

        foreach ($categories as $categoryData) {
            RestaurantCategory::firstOrCreate(
                ['restaurant_id' => $restaurant->id, 'name' => $categoryData['name']],
                [
                    'type' => $categoryData['type'],
                    'sort_order' => $categoryData['sort_order'],
                    'is_active' => true
                ]
            );
        }

        // Get global categories
        $breakfastCategory = Category::where('name', 'Breakfast')->first();
        $lunchCategory = Category::where('name', 'Lunch')->first();
        $dinnerCategory = Category::where('name', 'Dinner')->first();

        // Create global products if they don't exist
        $products = [
            [
                'category_id' => $lunchCategory?->id ?? 1,
                'name' => 'Chicken Curry',
                'slug' => 'chicken-curry',
                'description' => 'Traditional Sri Lankan chicken curry with aromatic spices',
                'is_active' => true
            ],
            [
                'category_id' => $lunchCategory?->id ?? 1,
                'name' => 'Fish Curry',
                'slug' => 'fish-curry',
                'description' => 'Fresh fish cooked in traditional Sri Lankan spices',
                'is_active' => true
            ],
            [
                'category_id' => $lunchCategory?->id ?? 1,
                'name' => 'Fried Rice',
                'slug' => 'fried-rice',
                'description' => 'Delicious fried rice with vegetables',
                'is_active' => true
            ],
            [
                'category_id' => $dinnerCategory?->id ?? 1,
                'name' => 'Chicken Kottu',
                'slug' => 'chicken-kottu',
                'description' => 'Spicy chicken kottu roti',
                'is_active' => true
            ],
        ];

        foreach ($products as $productData) {
            $product = Product::firstOrCreate(
                ['slug' => $productData['slug']],
                $productData
            );

            // Link product to restaurant with pricing
            DB::table('restaurant_products')->updateOrInsert(
                [
                    'restaurant_id' => $restaurant->id,
                    'product_id' => $product->id
                ],
                [
                    'price' => rand(400, 800),
                    'is_available' => true,
                    'is_featured' => rand(0, 1),
                    'preparation_time' => rand(15, 30),
                    'created_at' => now(),
                    'updated_at' => now()
                ]
            );
        }

        $this->command->info('Restaurant menu seeded successfully!');
    }
}
