<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <meta name="csrf-token" content="{{ csrf_token() }}">
    <title>Delivery Boy</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; background: #f5f5f5; overflow-x: hidden; }
        
        .login { height: 100vh; background: linear-gradient(135deg, #ff9800, #ff5722); display: flex; flex-direction: column; justify-content: center; align-items: center; padding: 20px; color: white; }
        .login-icon { font-size: 80px; margin-bottom: 10px; }
        .login h1 { font-size: 32px; margin-bottom: 40px; }
        .login-form { width: 100%; max-width: 400px; background: white; padding: 30px; border-radius: 16px; }
        .input { width: 100%; padding: 15px; margin-bottom: 15px; border: 1px solid #ddd; border-radius: 8px; font-size: 16px; }
        .btn-login { width: 100%; padding: 15px; background: #ff9800; color: white; border: none; border-radius: 8px; font-size: 18px; font-weight: bold; cursor: pointer; }
        
        .app { display: none; flex-direction: column; height: 100vh; }
        .header { background: linear-gradient(135deg, #ff9800, #ff5722); color: white; padding: 20px; box-shadow: 0 4px 12px rgba(0,0,0,0.15); }
        .header-top { display: flex; justify-content: space-between; align-items: center; }
        .btn-logout { background: rgba(255,255,255,0.2); color: white; border: none; padding: 10px 20px; border-radius: 20px; cursor: pointer; }
        .content { flex: 1; overflow-y: auto; padding: 15px 15px 80px; }
        .tab-content { display: none; }
        .tab-content.active { display: block; }
        
        .empty { text-align: center; padding: 60px 20px; color: #999; }
        .empty-icon { font-size: 60px; margin-bottom: 15px; }
        .btn { background: #ff9800; color: white; border: none; padding: 12px 30px; border-radius: 8px; cursor: pointer; }
        
        .list { display: flex; flex-direction: column; gap: 12px; }
        .card { background: white; border-radius: 16px; padding: 16px; box-shadow: 0 2px 8px rgba(0,0,0,0.08); cursor: pointer; transition: transform 0.2s; }
        .card:active { transform: scale(0.98); }
        .card-header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 12px; }
        .order-number { font-size: 18px; font-weight: bold; color: #333; }
        .order-amount { font-size: 20px; font-weight: bold; color: #4caf50; }
        .card-info { display: flex; align-items: center; gap: 8px; margin-bottom: 8px; color: #666; font-size: 14px; }
        .card-footer { display: flex; justify-content: space-between; align-items: center; margin-top: 12px; padding-top: 12px; border-top: 1px solid #f0f0f0; }
        .delivery-fee { color: #ff9800; font-weight: 600; }
        .distance-badge { background: #e3f2fd; color: #2196f3; padding: 4px 12px; border-radius: 12px; font-size: 12px; font-weight: 600; }
        
        .earnings-total { background: linear-gradient(135deg, #4caf50, #2e7d32); color: white; padding: 30px; border-radius: 16px; text-align: center; margin-bottom: 20px; }
        .earnings-total .amount { font-size: 36px; font-weight: bold; margin: 8px 0; }
        .earnings-list { display: flex; flex-direction: column; gap: 12px; }
        .earning-item { background: white; padding: 15px; border-radius: 12px; display: flex; align-items: center; gap: 15px; }
        .earning-item .icon { font-size: 32px; }
        .earning-item .value { font-size: 20px; font-weight: bold; }
        
        .nav { position: fixed; bottom: 0; left: 0; right: 0; background: white; display: flex; box-shadow: 0 -2px 10px rgba(0,0,0,0.1); }
        .nav button { flex: 1; display: flex; flex-direction: column; align-items: center; gap: 4px; background: none; border: none; color: #999; cursor: pointer; padding: 10px; }
        .nav button.active { color: #ff9800; }
        .nav-icon { font-size: 24px; }
        .nav-label { font-size: 12px; }
        
        .modal { display: none; position: fixed; top: 0; left: 0; right: 0; bottom: 0; background: rgba(0,0,0,0.6); align-items: flex-end; z-index: 1000; animation: fadeIn 0.3s; }
        .modal.show { display: flex; }
        @keyframes fadeIn { from { opacity: 0; } to { opacity: 1; } }
        @keyframes slideUp { from { transform: translateY(100%); } to { transform: translateY(0); } }
        .modal-box { background: white; width: 100%; max-height: 90vh; border-radius: 24px 24px 0 0; overflow-y: auto; animation: slideUp 0.3s; }
        .modal-header { padding: 20px; border-bottom: 1px solid #f0f0f0; display: flex; justify-content: space-between; align-items: center; }
        .modal-title { font-size: 20px; font-weight: bold; }
        .close { background: #f5f5f5; border: none; width: 32px; height: 32px; border-radius: 50%; font-size: 20px; cursor: pointer; }
        .modal-body { padding: 20px; }
        .detail-section { margin-bottom: 24px; }
        .section-title { font-size: 12px; color: #999; text-transform: uppercase; letter-spacing: 0.5px; margin-bottom: 8px; font-weight: 600; }
        .detail-card { background: #f8f9fa; padding: 16px; border-radius: 12px; }
        .detail-row { display: flex; justify-content: space-between; margin-bottom: 8px; }
        .detail-row:last-child { margin-bottom: 0; }
        .detail-label { color: #666; font-size: 14px; }
        .detail-value { font-weight: 600; color: #333; font-size: 14px; }
        .map-container { height: 200px; background: #e0e0e0; border-radius: 12px; overflow: hidden; margin-top: 12px; }
        .map-container iframe { width: 100%; height: 100%; border: 0; }
        .action-buttons { display: flex; gap: 10px; margin-top: 12px; }
        .btn-call, .btn-navigate { flex: 1; padding: 12px; border: none; border-radius: 12px; font-weight: 600; cursor: pointer; display: flex; align-items: center; justify-content: center; gap: 8px; }
        .btn-call { background: #2196f3; color: white; }
        .btn-navigate { background: #4caf50; color: white; }
        .modal-footer { padding: 20px; border-top: 1px solid #f0f0f0; }
        .btn-accept { width: 100%; padding: 16px; border: none; border-radius: 12px; font-size: 16px; font-weight: bold; cursor: pointer; color: white; background: linear-gradient(135deg, #4caf50, #2e7d32); }
        .btn-pickup { width: 100%; padding: 16px; border: none; border-radius: 12px; font-size: 16px; font-weight: bold; cursor: pointer; color: white; background: linear-gradient(135deg, #2196f3, #1976d2); }
        .btn-deliver { width: 100%; padding: 16px; border: none; border-radius: 12px; font-size: 16px; font-weight: bold; cursor: pointer; color: white; background: linear-gradient(135deg, #ff9800, #f57c00); }
        
        .badge { padding: 6px 12px; border-radius: 12px; font-size: 11px; font-weight: bold; text-transform: uppercase; }
        .badge-ready { background: #e8f5e9; color: #2e7d32; }
        .badge-picked_up { background: #e3f2fd; color: #1976d2; }
    </style>
</head>
<body>
    <div id="login" class="login">
        <div class="login-icon">🚴</div>
        <h1>Delivery Boy</h1>
        <div class="login-form">
            <input id="email" type="email" placeholder="Email" class="input" value="delivery@example.com">
            <input id="password" type="password" placeholder="Password" class="input" value="password">
            <button onclick="login()" class="btn-login">Login</button>
        </div>
    </div>

    <div id="app" class="app">
        <div class="header">
            <div class="header-top">
                <h2 id="header-title">Available Orders</h2>
                <button onclick="logout()" class="btn-logout">Logout</button>
            </div>
        </div>

        <div class="content">
            <div id="tab-0" class="tab-content active">
                <div id="available-empty" class="empty">
                    <div class="empty-icon">📦</div>
                    <p>No available orders</p>
                    <button onclick="loadAvailable()" class="btn">Refresh</button>
                </div>
                <div id="available-list" class="list" style="display:none;"></div>
            </div>

            <div id="tab-1" class="tab-content">
                <div style="margin-bottom: 15px; display: flex; gap: 10px;">
                    <select id="status-filter" onchange="filterMyOrders()" style="flex: 1; padding: 12px; border: 1px solid #ddd; border-radius: 8px; font-size: 14px; background: white;">
                        <option value="all">All Status</option>
                        <option value="assigned">Assigned</option>
                        <option value="ready">Ready</option>
                        <option value="picked_up">Picked Up</option>
                        <option value="delivered">Delivered</option>
                    </select>
                    <select id="date-filter" onchange="filterMyOrders()" style="flex: 1; padding: 12px; border: 1px solid #ddd; border-radius: 8px; font-size: 14px; background: white;">
                        <option value="all">All Time</option>
                        <option value="today">Today</option>
                        <option value="yesterday">Yesterday</option>
                        <option value="week">This Week</option>
                        <option value="month">This Month</option>
                    </select>
                </div>
                <div id="myorders-empty" class="empty">
                    <div class="empty-icon">🚚</div>
                    <p>No active deliveries</p>
                </div>
                <div id="myorders-list" class="list" style="display:none;"></div>
            </div>

            <div id="tab-2" class="tab-content">
                <div style="margin-bottom: 15px;">
                    <select id="earnings-filter" onchange="loadEarnings()" style="width: 100%; padding: 12px; border: 1px solid #ddd; border-radius: 8px; font-size: 14px; background: white;">
                        <option value="today">Today</option>
                        <option value="week">This Week</option>
                        <option value="month">This Month</option>
                        <option value="all">All Time</option>
                        <option value="custom">Custom Date</option>
                    </select>
                </div>
                <div id="custom-date-picker" style="display: none; margin-bottom: 15px; display: flex; gap: 10px;">
                    <input type="date" id="start-date" style="flex: 1; padding: 12px; border: 1px solid #ddd; border-radius: 8px; font-size: 14px;">
                    <input type="date" id="end-date" style="flex: 1; padding: 12px; border: 1px solid #ddd; border-radius: 8px; font-size: 14px;">
                    <button onclick="loadEarnings()" style="padding: 12px 20px; background: #4caf50; color: white; border: none; border-radius: 8px; font-weight: 600; cursor: pointer;">Go</button>
                </div>
                <div class="earnings-total">
                    <div id="earnings-period">Total Earnings</div>
                    <div class="amount" id="total-earnings">Rs. 0</div>
                    <div id="total-deliveries">0 Deliveries</div>
                </div>
                <div class="earnings-list">
                    <div class="earning-item">
                        <span class="icon">📅</span>
                        <div><div>Today</div><div class="value" id="today-earnings">Rs. 0</div></div>
                    </div>
                    <div class="earning-item">
                        <span class="icon">📊</span>
                        <div><div>This Week</div><div class="value" id="week-earnings">Rs. 0</div></div>
                    </div>
                    <div class="earning-item">
                        <span class="icon">📈</span>
                        <div><div>This Month</div><div class="value" id="month-earnings">Rs. 0</div></div>
                    </div>
                </div>
            </div>
        </div>

        <div class="nav">
            <button onclick="switchTab(0)" class="active"><span class="nav-icon">📋</span><span class="nav-label">Available</span></button>
            <button onclick="switchTab(1)"><span class="nav-icon">🚚</span><span class="nav-label">My Orders</span></button>
            <button onclick="switchTab(2)"><span class="nav-icon">💰</span><span class="nav-label">Earnings</span></button>
            <button onclick="window.location.href='/delivery-boy/profile'"><span class="nav-icon">👤</span><span class="nav-label">Profile</span></button>
        </div>
    </div>

    <div id="modal" class="modal" onclick="closeModal()">
        <div class="modal-box" onclick="event.stopPropagation()">
            <div class="modal-header">
                <h3 id="modal-title" class="modal-title">Order Details</h3>
                <button onclick="closeModal()" class="close">✕</button>
            </div>
            <div class="modal-body" id="modal-body"></div>
            <div class="modal-footer" id="modal-footer"></div>
        </div>
    </div>

    <script>
        let token = localStorage.getItem('delivery_token');
        let currentTab = 0;
        let selectedOrder = null;
        let allMyOrders = [];
        const tabs = ['Available Orders', 'My Orders', 'Earnings'];

        if (token) {
            document.getElementById('login').style.display = 'none';
            document.getElementById('app').style.display = 'flex';
            loadData();
        }

        async function login() {
            const email = document.getElementById('email').value;
            const password = document.getElementById('password').value;
            
            try {
                const res = await fetch('/api/login', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json', 'Accept': 'application/json' },
                    body: JSON.stringify({ email, password })
                });
                
                if (!res.ok) {
                    alert('Login failed: ' + res.status);
                    return;
                }
                
                const data = await res.json();
                
                if (data.token) {
                    token = data.token;
                    localStorage.setItem('delivery_token', token);
                    document.getElementById('login').style.display = 'none';
                    document.getElementById('app').style.display = 'flex';
                    loadData();
                } else {
                    alert('Login failed');
                }
            } catch (e) {
                alert('Error: ' + e.message);
            }
        }

        function logout() {
            localStorage.removeItem('delivery_token');
            location.reload();
        }

        function switchTab(index) {
            currentTab = index;
            document.querySelectorAll('.tab-content').forEach((el, i) => el.classList.toggle('active', i === index));
            document.querySelectorAll('.nav button').forEach((el, i) => el.classList.toggle('active', i === index));
            document.getElementById('header-title').textContent = tabs[index];
            
            if (index === 0) loadAvailable();
            if (index === 1) loadMyOrders();
            if (index === 2) loadEarnings();
        }

        async function loadData() {
            loadAvailable();
            loadMyOrders();
            loadEarnings();
        }

        async function loadAvailable() {
            try {
                const res = await fetch('/api/delivery-boy/orders/available', {
                    headers: { 'Authorization': 'Bearer ' + token, 'Accept': 'application/json' }
                });
                
                if (!res.ok) return;
                
                const data = await res.json();
                window.availableOrders = data.data || [];
                
                if (window.availableOrders.length === 0) {
                    document.getElementById('available-empty').style.display = 'block';
                    document.getElementById('available-list').style.display = 'none';
                } else {
                    document.getElementById('available-empty').style.display = 'none';
                    document.getElementById('available-list').style.display = 'flex';
                    document.getElementById('available-list').innerHTML = window.availableOrders.map((o, idx) => `
                        <div class="card" style="position: relative;">
                            <div onclick='viewOrderByIndex(${idx})'>
                                <div class="card-header">
                                    <span class="order-number">#${o.order_number}</span>
                                    <span class="order-amount">Rs. ${o.total_amount}</span>
                                </div>
                                <div class="card-info">🏪 ${o.restaurant?.name || ''}</div>
                                <div class="card-info">👤 ${o.user?.name || ''}</div>
                                <div class="card-info">📍 ${o.delivery_address}</div>
                                <div class="card-footer">
                                    <span class="delivery-fee">💰 Rs. ${o.delivery_fee}</span>
                                    <span class="distance-badge">📍 ${o.distance || o.distance_km || '0'} km</span>
                                </div>
                            </div>
                            <button onclick="event.stopPropagation(); viewMapByIndex(${idx})" style="width: 100%; margin-top: 12px; background: #2196f3; color: white; border: none; padding: 10px; border-radius: 8px; font-size: 14px; font-weight: 600; cursor: pointer;">🗺️ View Map</button>
                        </div>
                    `).join('');
                }
            } catch (e) {}
        }

        async function loadMyOrders() {
            try {
                const res = await fetch('/api/delivery-boy/orders/my-orders', {
                    headers: { 'Authorization': 'Bearer ' + token, 'Accept': 'application/json' }
                });
                
                if (!res.ok) return;
                
                const data = await res.json();
                allMyOrders = data.data || [];
                filterMyOrders();
            } catch (e) {}
        }

        function filterMyOrders() {
            const statusFilter = document.getElementById('status-filter')?.value || 'all';
            const dateFilter = document.getElementById('date-filter')?.value || 'all';
            
            let orders = allMyOrders;
            
            // Filter by status
            if (statusFilter !== 'all') {
                orders = orders.filter(o => o.status === statusFilter);
            }
            
            // Filter by date
            if (dateFilter !== 'all') {
                const now = new Date();
                const today = new Date(now.getFullYear(), now.getMonth(), now.getDate());
                
                orders = orders.filter(o => {
                    const orderDate = new Date(o.created_at);
                    
                    if (dateFilter === 'today') {
                        return orderDate >= today;
                    } else if (dateFilter === 'yesterday') {
                        const yesterday = new Date(today);
                        yesterday.setDate(yesterday.getDate() - 1);
                        return orderDate >= yesterday && orderDate < today;
                    } else if (dateFilter === 'week') {
                        const weekAgo = new Date(today);
                        weekAgo.setDate(weekAgo.getDate() - 7);
                        return orderDate >= weekAgo;
                    } else if (dateFilter === 'month') {
                        const monthAgo = new Date(today);
                        monthAgo.setMonth(monthAgo.getMonth() - 1);
                        return orderDate >= monthAgo;
                    }
                    return true;
                });
            }
            
            if (orders.length === 0) {
                document.getElementById('myorders-empty').style.display = 'block';
                document.getElementById('myorders-list').style.display = 'none';
            } else {
                document.getElementById('myorders-empty').style.display = 'none';
                document.getElementById('myorders-list').style.display = 'flex';
                document.getElementById('myorders-list').innerHTML = orders.map(o => {
                    const dateStr = o.created_at || o.updated_at;
                    let dateDisplay = '';
                    if (dateStr) {
                        try {
                            const d = new Date(dateStr);
                            if (!isNaN(d.getTime())) {
                                dateDisplay = `<div class="card-info" style="color: #999; font-size: 12px;">📅 ${d.toLocaleDateString()} ${d.toLocaleTimeString()}</div>`;
                            }
                        } catch (e) {}
                    }
                    return `
                        <div class="card" onclick='viewOrder(${JSON.stringify(o).replace(/'/g, "&apos;")})'>
                            <div class="card-header">
                                <span class="order-number">#${o.order_number}</span>
                                <span class="badge badge-${o.status}">${o.status}</span>
                            </div>
                            <div class="card-info">👤 ${o.user?.name || ''}</div>
                            <div class="card-info">📞 ${o.user?.phone || ''}</div>
                            <div class="card-info">📍 ${o.delivery_address}</div>
                            ${dateDisplay}
                        </div>
                    `;
                }).join('');
            }
        }

        async function loadEarnings() {
            const filter = document.getElementById('earnings-filter')?.value || 'all';
            const customDatePicker = document.getElementById('custom-date-picker');
            
            if (filter === 'custom') {
                customDatePicker.style.display = 'flex';
            } else {
                customDatePicker.style.display = 'none';
            }
            
            let url = '/api/delivery-boy/earnings/summary';
            const params = new URLSearchParams();
            
            if (filter === 'custom') {
                const startDate = document.getElementById('start-date')?.value;
                const endDate = document.getElementById('end-date')?.value;
                if (startDate && endDate) {
                    params.append('start_date', startDate);
                    params.append('end_date', endDate);
                } else {
                    return;
                }
            } else if (filter !== 'all') {
                params.append('period', filter);
            }
            
            if (params.toString()) {
                url += '?' + params.toString();
            }
            
            try {
                const res = await fetch(url, {
                    headers: { 'Authorization': 'Bearer ' + token, 'Accept': 'application/json' }
                });
                
                if (!res.ok) return;
                
                const data = await res.json();
                
                const periodLabels = {
                    'today': 'Today',
                    'week': 'This Week',
                    'month': 'This Month',
                    'all': 'Total Earnings',
                    'custom': 'Custom Period'
                };
                
                document.getElementById('earnings-period').textContent = periodLabels[filter] || 'Total Earnings';
                document.getElementById('total-earnings').textContent = 'Rs. ' + (data.total || 0);
                document.getElementById('total-deliveries').textContent = (data.deliveries || 0) + ' Deliveries';
                document.getElementById('today-earnings').textContent = 'Rs. ' + (data.today || 0);
                document.getElementById('week-earnings').textContent = 'Rs. ' + (data.week || 0);
                document.getElementById('month-earnings').textContent = 'Rs. ' + (data.month || 0);
            } catch (e) {}
        }

        function viewOrder(order) {
            selectedOrder = order;
            document.getElementById('modal-title').textContent = 'Order #' + order.order_number;
            
            const paymentMethod = order.payment_method === 'cash' ? '💵 Cash on Delivery' : '💳 Card Payment';
            const itemsHtml = order.items?.map(item => `
                <div class="detail-row">
                    <span class="detail-label">${item.quantity}x ${item.product_name}</span>
                    <span class="detail-value">Rs. ${item.total_price || (item.quantity * item.unit_price)}</span>
                </div>
            `).join('') || '<p>No items</p>';
            
            document.getElementById('modal-body').innerHTML = `
                <div class="detail-section">
                    <div class="section-title">Restaurant</div>
                    <div class="detail-card">
                        <div class="detail-row">
                            <span class="detail-label">🏪 Name</span>
                            <span class="detail-value">${order.restaurant?.name || 'N/A'}</span>
                        </div>
                    </div>
                </div>
                <div class="detail-section">
                    <div class="section-title">Customer</div>
                    <div class="detail-card">
                        <div class="detail-row">
                            <span class="detail-label">👤 Name</span>
                            <span class="detail-value">${order.user?.name || 'N/A'}</span>
                        </div>
                        <div class="detail-row">
                            <span class="detail-label">📞 Phone</span>
                            <span class="detail-value">${order.user?.phone || order.delivery_phone || 'N/A'}</span>
                        </div>
                    </div>
                    <div class="action-buttons">
                        <button onclick="callCustomer()" class="btn-call">📞 Call</button>
                        <button onclick="openMap()" class="btn-navigate">🗺️ Navigate</button>
                    </div>
                </div>
                <div class="detail-section">
                    <div class="section-title">Order Items</div>
                    <div class="detail-card">
                        ${itemsHtml}
                    </div>
                </div>
                <div class="detail-section">
                    <div class="section-title">Delivery Address</div>
                    <div class="detail-card">
                        <div class="detail-value">${order.delivery_address}</div>
                    </div>
                    ${order.delivery_latitude && order.delivery_longitude ? `
                        <div class="map-container">
                            <iframe src="https://www.google.com/maps?q=${order.delivery_latitude},${order.delivery_longitude}&output=embed" allowfullscreen loading="lazy"></iframe>
                        </div>
                    ` : ''}
                </div>
                <div class="detail-section">
                    <div class="section-title">Payment</div>
                    <div class="detail-card">
                        <div class="detail-row">
                            <span class="detail-label">Payment Method</span>
                            <span class="detail-value">${paymentMethod}</span>
                        </div>
                        <div class="detail-row">
                            <span class="detail-label">Total Amount</span>
                            <span class="detail-value" style="color:#4caf50; font-size:18px;">Rs. ${order.total_amount || order.total || 0}</span>
                        </div>
                        <div class="detail-row">
                            <span class="detail-label">Delivery Fee</span>
                            <span class="detail-value" style="color:#ff9800">Rs. ${order.delivery_fee || 0}</span>
                        </div>
                        <div class="detail-row">
                            <span class="detail-label">Distance</span>
                            <span class="detail-value">${order.distance || order.distance_km || '0'} km</span>
                        </div>
                    </div>
                </div>
            `;
            
            let footer = '';
            if (!order.delivery && currentTab === 0) {
                footer = '<button onclick="acceptOrder()" class="btn-accept">✓ Accept Order</button>';
            } else if (order.status === 'assigned' || order.status === 'ready') {
                footer = '<button onclick="updateStatus(\'picked_up\')" class="btn-pickup">📦 Pick Up Order</button>';
            } else if (order.status === 'picked_up') {
                footer = '<button onclick="updateStatus(\'delivered\')" class="btn-deliver">✓ Mark as Delivered</button>';
            }
            document.getElementById('modal-footer').innerHTML = footer;
            
            document.getElementById('modal').classList.add('show');
        }

        function closeModal() {
            document.getElementById('modal').classList.remove('show');
        }

        function callCustomer() {
            const phone = selectedOrder.user?.phone || selectedOrder.delivery_phone;
            if (phone) window.location.href = 'tel:' + phone;
        }

        async function acceptOrder() {
            if (!confirm('Are you sure you want to accept this order?\n\nOrder: #' + selectedOrder.order_number + '\nDelivery Fee: Rs. ' + selectedOrder.delivery_fee)) {
                return;
            }
            
            try {
                const res = await fetch('/api/delivery-boy/orders/' + selectedOrder.id + '/accept', {
                    method: 'POST',
                    headers: { 'Authorization': 'Bearer ' + token, 'Accept': 'application/json', 'Content-Type': 'application/json' }
                });
                
                if (res.ok) {
                    alert('Order accepted!');
                    closeModal();
                    loadData();
                } else {
                    const error = await res.json();
                    alert('Failed: ' + (error.message || 'Unknown error'));
                }
            } catch (e) {
                alert('Error: ' + e.message);
            }
        }

        async function updateStatus(status) {
            let confirmMsg = '';
            if (status === 'picked_up') {
                confirmMsg = 'Confirm that you have picked up this order?\n\nOrder: #' + selectedOrder.order_number + '\nRestaurant: ' + (selectedOrder.restaurant?.name || 'N/A');
            } else if (status === 'delivered') {
                const isCashPayment = selectedOrder.payment_method === 'cash';
                const amount = selectedOrder.total_amount || selectedOrder.total || 0;
                
                if (isCashPayment) {
                    confirmMsg = 'Have you collected the payment from customer?\n\n💵 Cash Payment: Rs. ' + amount + '\n\nOrder: #' + selectedOrder.order_number + '\nCustomer: ' + (selectedOrder.user?.name || 'N/A');
                } else {
                    confirmMsg = 'Confirm that you have delivered this order?\n\n💳 Payment: Already Paid Online\n\nOrder: #' + selectedOrder.order_number + '\nCustomer: ' + (selectedOrder.user?.name || 'N/A');
                }
            }
            
            if (confirmMsg && !confirm(confirmMsg)) {
                return;
            }
            
            try {
                const res = await fetch('/api/delivery-boy/deliveries/' + selectedOrder.id + '/status', {
                    method: 'PUT',
                    headers: { 'Authorization': 'Bearer ' + token, 'Accept': 'application/json', 'Content-Type': 'application/json' },
                    body: JSON.stringify({ status })
                });
                
                if (res.ok) {
                    alert('Status updated!');
                    closeModal();
                    loadData();
                } else {
                    alert('Failed to update status');
                }
            } catch (e) {
                alert('Error: ' + e.message);
            }
        }

        function openMap() {
            const lat = selectedOrder.delivery_latitude;
            const lng = selectedOrder.delivery_longitude;
            if (lat && lng) {
                window.open(`https://www.google.com/maps/dir/?api=1&destination=${lat},${lng}`);
            }
        }

        function viewMapOnly(order) {
            selectedOrder = order;
            document.getElementById('modal-title').textContent = 'Map - Order #' + order.order_number;
            const restLat = parseFloat(order.restaurant?.latitude) || 0;
            const restLng = parseFloat(order.restaurant?.longitude) || 0;
            const custLat = parseFloat(order.delivery_latitude) || 0;
            const custLng = parseFloat(order.delivery_longitude) || 0;
            
            document.getElementById('modal-body').innerHTML = `
                <div class="detail-section">
                    <div class="section-title">Delivery Route</div>
                    ${custLat && custLng && restLat && restLng ? `
                        <div id="routeMap" style="height: 400px; background: #e0e0e0; border-radius: 12px; overflow: hidden;"></div>
                        <a href="https://www.google.com/maps/dir/?api=1&origin=${restLat},${restLng}&destination=${custLat},${custLng}&travelmode=driving" target="_blank" style="display: block; margin-top: 12px; background: linear-gradient(135deg, #4caf50, #2e7d32); color: white; padding: 14px; border-radius: 12px; text-align: center; text-decoration: none; font-weight: bold; font-size: 16px;">
                            🗺️ Open Directions in Google Maps
                        </a>
                    ` : '<p>Map not available</p>'}
                    <div class="detail-card" style="margin-top: 12px;">
                        <div class="detail-row">
                            <span class="detail-label">Distance</span>
                            <span class="detail-value">${order.distance || order.distance_km || '0'} km</span>
                        </div>
                        <div class="detail-row">
                            <span class="detail-label">Address</span>
                            <span class="detail-value">${order.delivery_address}</span>
                        </div>
                    </div>
                </div>
            `;
            document.getElementById('modal-footer').innerHTML = '<button onclick="closeModal()" class="btn-accept">Close</button>';
            document.getElementById('modal').classList.add('show');
            
            if (custLat && custLng && restLat && restLng) {
                setTimeout(() => initRouteMap(restLat, restLng, custLat, custLng), 100);
            }
        }

        function initRouteMap(restLat, restLng, custLat, custLng) {
            if (!window.google) {
                loadGoogleMapsScript(() => initRouteMap(restLat, restLng, custLat, custLng));
                return;
            }
            
            const map = new google.maps.Map(document.getElementById('routeMap'), {
                center: { lat: restLat, lng: restLng },
                zoom: 13
            });
            
            const directionsService = new google.maps.DirectionsService();
            const directionsRenderer = new google.maps.DirectionsRenderer({
                map: map,
                polylineOptions: {
                    strokeColor: '#FF6B35',
                    strokeWeight: 5
                }
            });
            
            directionsService.route({
                origin: { lat: restLat, lng: restLng },
                destination: { lat: custLat, lng: custLng },
                travelMode: google.maps.TravelMode.WALKING
            }, (result, status) => {
                if (status === 'OK') {
                    directionsRenderer.setDirections(result);
                }
            });
        }

        function loadGoogleMapsScript(callback) {
            if (window.google) {
                callback();
                return;
            }
            const script = document.createElement('script');
            script.src = 'https://maps.googleapis.com/maps/api/js?key=AIzaSyBrfy7IL9up-52b6vB0DyQdMTZYAJPDOuw';
            script.async = true;
            script.onload = callback;
            document.head.appendChild(script);
        }

        function viewOrderByIndex(idx) {
            viewOrder(window.availableOrders[idx]);
        }

        function viewMapByIndex(idx) {
            viewMapOnly(window.availableOrders[idx]);
        }
    </script>
</body>
</html>
